from time import sleep, time
import threading
from ambient_api.ambientapi import AmbientAPI
from lib.utils import Utils

class AmbientLib(Utils):
    def __init__(self, ambient_device, api_key, application_key, seconds_request, log_id= "AMBIENT") -> None:
        super().__init__()  # Inicializa primero la clase base
        self.log_id = log_id
        self.device_name = ambient_device
        self.api_key = api_key
        self.application_key = application_key
        self.seconds_request = seconds_request
        self.weather = None
        self.last_ambient_timestamp = time()
        self.last_data = None
        threading.Thread(target=self.connect, daemon=True).start()

    def connect(self) -> None:
        """
        Intenta establecer una conexión con el dispositivo ambient weather especificado.
        """
        self.log("Conectando a dispositivo ambient weather")
        try:
            # Inicializar la API de Ambient Weather
            self.log("Inicializando la API de Ambient Weather")
            self.weather = AmbientAPI(
                AMBIENT_LOG_LEVEL='debug',
                AMBIENT_API_KEY= self.api_key,
                AMBIENT_APPLICATION_KEY= self.application_key
            )
            self.read()
        except Exception as e:
            self.log(f"Error al conectar con el dispositivo ambiental: {e}")

    def read(self) -> None:
        """
        Continuously reads data from the specified ambient device.

        Args:
            AMBIENT_DEVICE (str): The identifier of the ambient device to read.
            WEATHER: An instance of the Ambient Weather API.
        """
        self.log("Reading data from ambient device")
        while True:
            try:
                # Obtener los dispositivos disponibles
                devices = self.weather.get_devices()

                if not devices:
                    self.log("No devices found. Retrying...")
                    sleep(30)
                    continue

                # Buscar el dispositivo especificado
                for device in devices:
                    if self.device_name in str(device):
                        # Extraer y registrar los datos del dispositivo
                        data = device.last_data
                        print(data)
                        date = int(data["dateutc"]/1000)
                        if time()- date <= 900:
                            if data != self.last_data:
                                dict_data = self.process_data(data)
                                if dict_data:
                                    data = {
                                        "name_id": "AMBIENT",
                                        "data": dict_data
                                    }
                                    self.log(f"[AMBIENT Data Emitted]: {data['name_id']}, {data['data']}")
                                    # Emite los datos
                                    self.emit(data_type=self.log_id, data=data)
                                    print(f"Sent data to {self.server_address}: {data}")
                                    self.last_ambient_timestamp = time()
                                    self.last_data = data
                                    break
                        else:
                            self.log("Data too old")
                            break

                sleep(self.seconds_request)

            except Exception as e:
                self.log(f"Error reading ambient device: {e}")

            sleep(1)
    
    def process_data(self, data):
        try:
            dict_data = {
                "wind_speed": self.mph2ms(data["windspeedmph"]),
                "wind_direction": data["winddir"],
                "wind_gust": data["windgustmph"],
                "temperature": self.fahrenheit2celsius(data["tempf"]),
                "humidity": data["humidity"],
                "uvindex": data["uv"],
                "solarrad": data["solarradiation"],
                "rain": data["hourlyrainin"] if "hourlyrainin" in data else 0,
            }
            return dict_data
        
        except Exception as e:
            self.log(f"Error processing data: {e}")
            return None
            
    def fahrenheit2celsius(self, tempf):
        return round((tempf - 32.0) * (5.0/9.0), 1) # Preserve 1 decimal only

    def mph2ms(self, mph):
        return round(mph * 0.44704, 1) # Preserve 1 decimal only
