import io
import pynmea2
import threading
import json
import math
import threading
from serial import Serial, SerialException
from time import sleep, time
from lib.utils import Utils
from datetime import datetime

from shapely.geometry import Point
from shapely.geometry.polygon import Polygon

#

class GpsLib(Utils):

	def __init__(self, usbdevnode, baudrate = 9600, path_geofence = "/srv/datalogger_mmr/geofence.json" , log_id="GPS_USB"):
		self.baudrate = baudrate
		self.log_id = log_id
		self.usbdevnode = usbdevnode
		self.geofence = self.update_geofence(path_geofence)
		self.last_timestamp = time()
		self.machine_id = self.get_product_id()
		self.update_reg_name()

		# Real time status variables
		self.lat = 0
		self.lon = 0
		self.last_lat = 0
		self.last_lon = 0
		self.angle = 0
		self.speed = 0
		self.text_gprmc = None
		self.text_gpgga = None
		self.timestamp = 0
		self.datetime = 0

		# Distance calculation variables
		self.a = 6378137  # meters
		self.f = 1 / 298.257223563
		self.b = 6356752.314245  # meters; b = (1 - f)a
		self.MILES_PER_KILOMETER = 0.621371
		self.MAX_ITERATIONS = 200
		self.CONVERGENCE_THRESHOLD = 1e-12  # .000,000,000,001
		self.distance_from_last_point = 0   # Also used for match()ing.

		self.connect()
		self.log("GpsSerial USB listener init [%s @ %s bps]" %(self.usbdevnode.get_devnode(), self.baudrate))
		threading.Thread(target = self.read).start()


	def update_reg_name(self):
		self.reg_number = ""
		try:
			location_assigned = self.get_location_assigned()
			if location_assigned:
				_, reg_number = location_assigned.split("-")
				self.reg_number = reg_number
		except:
			self.traceback()
		

	def update_geofence(self, file_path):
		geofence = []
		try: 
			with open(file_path) as file:
				data = json.load(file)
			for subdata in data["features"]:
				#print(subdata["geometry"]["coordinates"][0])
				geofence.append({
					"geofence_id": subdata["properties"]["id"],
					"polygon": Polygon(subdata["geometry"]["coordinates"][0]),
					"road_type": subdata["properties"]["tipocamino"],
					"location": subdata["properties"]["ubicacion"]
				})
			return geofence
		except Exception as ex:
			self.traceback()
			return None
		
	def get_data_from_geofence(self, lat, lon):
		point = Point(lon, lat)
		#print(self.geofence)
		if self.geofence:
			for zone in self.geofence:
				if zone["polygon"].contains(point):
					dict_data = zone.copy()
					dict_data.pop("polygon")
					return dict_data
		dict_data = self.get_default_geofence_values()
		return dict_data
	
	def get_default_geofence_values(self):
		return {
			"geofence_id": -1,
			"road_type": "UNKNOWN",
			"location": "UNKNOWN"
		}
	
	def report_not_fix(self):
		data = {
			'latitude': 0,
            'longitude': 0,
            'last_latitude': 0,
            'last_longitude': 0,
            'speed': 0,
            'distance': 0,
            'nmea': None,
            'datetime': self.datetime,
            'angle': 0,
            "timestamp": self.timestamp
		}
		return data

	def get_current_variables(self):
		data =  {
			'latitude': self.lat,
			'longitude': self.lon,
			'last_latitude': self.last_lat,
			'last_longitude': self.last_lon,
			'speed': round(self.speed, 1),
			'distance': round(self.distance, 1),
			'nmea': self.text_gprmc,
			'datetime': self.datetime,
			'angle': round(self.angle, 1),
			"timestamp": self.timestamp
				}
		return data

	def connect(self) -> None:
		"""
		This function attempts to establish a serial connection with the specified USB device node.

		"""

		try:
			self.log(f"Try to connect serial port: {self.usbdevnode.get_devnode()}")
			self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=1)
			self.sio = io.TextIOWrapper(io.BufferedRWPair(self.serial_module, self.serial_module))
			self.log("GpsSerial initialized [%s]" % self.serial_module)

		except Exception as Ex:
			self.log(Ex)
	


	def read(self) -> None:
		"""
		This function continuously reads lines from the serial module and processes them.
		If the line is empty, the function skips it.

		"""

		self.log("Reading line from serial")
		while True:
			if self.serial_module is None:
				self.connect()
			else:
				line = self.sio.readline()
				if not line: break
				self.process_line(line)
		sleep(0.1)

	def process_line(self, data):
		try:			
			data = data.replace("\n", "")
			if "RMC" in data:
				self.log("GOT [%s]" % data)
				result = self.decode_rmc(data)
				if result is not None:
					self.write_file("/gps_live.json", json.dumps(result), "w")
					# Check if the point is inside a zone
					geofence_data= self.get_data_from_geofence(result["latitude"], result["longitude"])
					dict_data = {
						'latitude': result["latitude"],
						'longitude': result["longitude"],
						'speed_kmh': result["speed"],
						'distance': result["distance"],
						'angle': result["angle"],
						'timestamp': result["timestamp"]
					}
					dict_data.update(geofence_data)
					self.log(f"DICT DATA: {dict_data}")
					self.emit(data_type=self.log_id, data= dict_data)
					tablet_data = {
						"name": self.reg_number,
						"vehicle": int(self.machine_id),
						"lat": result["latitude"],
						"lon": result["longitude"]
					}
					self.write_file("/var/www/html/data.json", json.dumps(tablet_data), "w")
				self.write_file("/NMEA.txt", "%s | %s\n" % (self.get_datetime(), json.dumps(data)), "a")

			if "GGA" in data:
				self.write_file("/NMEA.txt", "%s | %s\n" % (self.get_datetime(), json.dumps(data)), "a")

		except Exception as ex:
			self.traceback()
	
	def decode_rmc(self, line):

		try:
			self.text_gprmc = line
			line = pynmea2.parse(line)

			# 1. Update last known position
			self.last_lat = self.lat
			self.last_lon = self.lon

			# 2. Update current position
			self.lat = round(line.latitude, 6)
			self.lon = round(line.longitude, 6)

			# 3. Calculate angle
			self.angle = math.atan2(self.lon - self.last_lon, self.lat - self.last_lat) * 180/3.141592
			if self.angle < 0:
				self.angle = self.angle + 360

			# 4. Calculate speed
			self.distance = self.vincentyInverse((self.last_lat, self.last_lon), (self.lat, self.lon))
			self.speed = line.spd_over_grnd

			# 5. Update timestamp and datetime
			now = datetime.now()
			self.timestamp = int(datetime.timestamp(now))
			self.datetime = now.strftime("%Y-%m-%d %H:%M:%S")

			data = self.get_current_variables()
			self.last_timestamp = time()
			self.log("decodeGPRMC(): %s" % data)
			return data

		except ValueError as exc:
			self.log("GPS NOFIX: Unable to find Lat/Lon in NMEA frame [%s]" % line)
			return self.report_not_fix()
		except Exception as ex:
			self.traceback()
			return None

	


	# This is calculate to eye3
	# This is Vincenty's inverse, sourced from https://pypi.python.org/pypi/vincenty/0.1.4
	def vincentyInverse(self, point1, point2, miles=False):
		# short-circuit coincident points
		if point1[0] == point2[0] and point1[1] == point2[1]:
			return 0.0

		U1 = math.atan((1 - self.f) * math.tan(math.radians(point1[0])))
		U2 = math.atan((1 - self.f) * math.tan(math.radians(point2[0])))
		L = math.radians(point2[1] - point1[1])
		Lambda = L

		sinU1 = math.sin(U1)
		cosU1 = math.cos(U1)
		sinU2 = math.sin(U2)
		cosU2 = math.cos(U2)

		for iteration in range(self.MAX_ITERATIONS):
			sinLambda = math.sin(Lambda)
			cosLambda = math.cos(Lambda)
			sinSigma = math.sqrt((cosU2 * sinLambda) ** 2 + (cosU1 * sinU2 - sinU1 * cosU2 * cosLambda) ** 2)

			if sinSigma == 0:
				return 0.0 # coincident points

			cosSigma = sinU1 * sinU2 + cosU1 * cosU2 * cosLambda
			sigma = math.atan2(sinSigma, cosSigma)
			sinAlpha = cosU1 * cosU2 * sinLambda / sinSigma
			cosSqAlpha = 1 - sinAlpha ** 2

			try:
				cos2SigmaM = cosSigma - 2 * sinU1 * sinU2 / cosSqAlpha
			except ZeroDivisionError:
				cos2SigmaM = 0

			C = self.f / 16 * cosSqAlpha * (4 + self.f * (4 - 3 * cosSqAlpha))
			LambdaPrev = Lambda
			Lambda = L + (1 - C) * self.f * sinAlpha * (sigma + C * sinSigma * (cos2SigmaM + C * cosSigma * (-1 + 2 * cos2SigmaM ** 2)))
			if abs(Lambda - LambdaPrev) < self.CONVERGENCE_THRESHOLD:
				break     # Successful convergence
		else:
			return None   # failure to converge

		uSq = cosSqAlpha * (self.a ** 2 - self.b ** 2) / (self.b ** 2)
		A = 1 + uSq / 16384 * (4096 + uSq * (-768 + uSq * (320 - 175 * uSq)))
		B = uSq / 1024 * (256 + uSq * (-128 + uSq * (74 - 47 * uSq)))

		deltaSigma = B * sinSigma * (cos2SigmaM + B / 4 * (cosSigma * (-1 + 2 * cos2SigmaM ** 2) - B / 6 * cos2SigmaM * (-3 + 4 * sinSigma ** 2) * (-3 + 4 * cos2SigmaM ** 2)))
		s = self.b * A * (sigma - deltaSigma)
		s /= 1000 # meters to kilometers
		if miles:
			s *= self.MILES_PER_KILOMETER
			return round(s, 6)       # miles
		else:
			return round(s*1000, 6)  # meters






