import threading
import json
from serial import Serial, serialutil
from time import time, sleep
from lib.utils import Utils

class SerialLib(Utils):
    def __init__(self, usbdevnode, level_curve, model_sensor_level, max_cistern_level, baudrate: int = 38400, log_id = "SERIAL") -> None:
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.usbdevnode = usbdevnode
        self.v_min, self.v_max = level_curve # Lista con min and max voltage signal: [v_min, v_max]
        self.model_sensor_level = model_sensor_level #TUF or SATEL
        self.max_cistern_level = max_cistern_level # Max nivel de estanque del aljibe
        self.acum_level = []
        self.seconds_mean_level= 10
        self.timer_level= serialutil.Timeout(self.seconds_mean_level)
        self.last_valve_timestamp = time()
        
        threading.Thread(target = self.connect).start()

    def connect(self) -> None:
        """
        This function attempts to establish a serial connection with the specified USB device node.

        """

        try:
            self.log(f"Try to connect serial port: {self.usbdevnode.get_devnode()}")
            self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=self.timeout)
            self.read()

        except Exception as Ex:
            self.log(Ex)

    def read(self) -> None:
        """
        This function continuously reads lines from the serial module and processes them.
        If the line is empty, the function skips it.

        """
        # modo:0;extra:0;ltr:0;ltc:0;rtr:0;rtc:0;m_extra:1;m_ltr:0;m_ltc:1;m_rtr:1;m_rtc:1;a_extra:1;a_ltr:0;a_ltc:1;a_rtr:1;a_rtc:1;templm:0.0
        self.log("Reading line from serial")
        while True:
            try:
                raw_line = self.serial_module.readline().decode("utf-8")
                line = raw_line.strip()
                if line =="":
                    pass
                else:
                    self.log(f"GOT: {line}")
                    if "modo" in line:self.process_line(line)
                sleep(0.01)
            except:
                self.log("Error decoding line")

    def process_line(self, line: str) -> None:
        try:
            # 1. Process data 
            data_dict = {}
            data_split = line.split(';')
            for data in data_split:
                if not data: continue
                data = data.upper()
                if "M_" in data and not "M_EXTRA" in data: 
                    self.log(data)
                    serial_var, serial_value = data.split(':')
                    data_dict[serial_var] = int(float(serial_value))

                if "EXTRA_0" in data or "EXTRA_1" in data or "EXTRA_2" in data: 
                    self.log(data)
                    serial_var, serial_value = data.split(':')
                    data_dict[serial_var] = float(serial_value)

            # 2. Construct Dict
            valve_state = {
                "valve_state": [data_dict["M_LTR"], data_dict["M_LTC"], data_dict["M_RTC"], data_dict["M_RTR"]],
                "m_ltr": data_dict["M_LTR"],
                "m_ltc": data_dict["M_LTC"],
                "m_rtc": data_dict["M_RTC"],
                "m_rtr": data_dict["M_RTR"],
                "timestamp": time()
            }

            try:
                if "EXTRA_0" in data_dict and "EXTRA_1" in data_dict and "EXTRA_2" in data_dict:
                    valve_state["extra_0"] = data_dict["EXTRA_0"]
                    valve_state["extra_1"] = data_dict["EXTRA_1"]
                    valve_state["extra_2"] = data_dict["EXTRA_2"]
                
                # 3. Añadir data nivel de estanque de aljibe, solo si se ha ingresado la calibración
                if  self.v_min != 0 or self.v_max != 0 and self.model_sensor_level == "SATEL": 

                    # 3.1 Calcular porcentaje de nivel del estanque
                    level = int(((data_dict["EXTRA_1"] - self.v_min) / (self.v_max - self.v_min)) * 100)  # Valor en porcentaje

                    # 3.2 Añadir al diccionario y truncar en caso de que este fuera de los rangos 0 <= level <= 100
                    if 0 <= level <= 100: 
                        valve_state["level"] = level
                    elif level < 0: 
                        valve_state["level"] = 0 # Truncar en 0 si el nivel da negativo
                    else: 
                        valve_state["level"] = 100 # Truncar en 100 si el nivel es mayor a 100
                    
                    self.acum_level.append(valve_state["level"])
                    self.log(f"NIVEL_ESTANQUE: {level}%")

                    # 3.3 Save mean_data to use in map web every 3 seconds
                    if self.timer_level.expired():
                        # 3.3.1 Calculate mean
                        print(f"Acumulador: {self.acum_level}")
                        mean_level = int(sum(self.acum_level) / len(self.acum_level))

                        # 3.3.2 Restart aux var and timer
                        self.timer_level.restart(self.seconds_mean_level)
                        self.acum_level = []

                        # 3.3.3 Save data
                        if mean_level>= 0:
                            level_data = {
                                "porcentaje": mean_level,
                                "volumen_estanque": int(self.max_cistern_level*mean_level/100), 
                                "timestamp": int(time())
                            }
                            self.log(level_data)
                            self.write_file("/var/www/html/nivel.json", json.dumps(level_data), "w")

            except Exception as ex:
                print(f"Error process level data: {ex}")


            # 4. Send data to local server
            self.log(valve_state) 
            self.emit(data_type=self.log_id, data=valve_state)
            self.last_valve_timestamp = time()
            
        except:
            self.traceback() 
