/**
 * file       TinyGsmClientA6.h
 * author     Volodymyr Shymanskyy
 * license    LGPL-3.0
 * copyright  Copyright (c) 2016 Volodymyr Shymanskyy
 * date       Nov 2016
 */

#ifndef TinyGsmClientA6_h
#define TinyGsmClientA6_h

//#define TINY_GSM_DEBUG Serial

#if !defined(TINY_GSM_RX_BUFFER)
#define TINY_GSM_RX_BUFFER 256
#endif

#define TINY_GSM_MUX_COUNT 8

#include "TinyGsmCommon.h"

#define GSM_NL "\r\n"
static const char GSM_OK[] TINY_GSM_PROGMEM = "OK" GSM_NL;
static const char GSM_ERROR[] TINY_GSM_PROGMEM = "ERROR" GSM_NL;

enum SimStatus {
	SIM_ERROR = 0,
	SIM_READY = 1,
	SIM_LOCKED = 2,
};

enum RegStatus {
	REG_UNREGISTERED = 0,
	REG_SEARCHING    = 2,
	REG_DENIED       = 3,
	REG_OK_HOME      = 1,
	REG_OK_ROAMING   = 5,
	REG_UNKNOWN      = 4,
};


class TinyGsm
{

public:

	class GsmClient : public Client
	{
		friend class TinyGsm;
		typedef TinyGsmFifo<uint8_t, TINY_GSM_RX_BUFFER> RxFifo;

	public:
		GsmClient() {}

		GsmClient(TinyGsm& modem)
		{
			init(&modem);
		}

		bool init(TinyGsm* modem)
		{
			this->at = modem;
			this->mux = -1;
			sock_connected = false;

			return true;
		}

	public:
		virtual int connect(const char *host, uint16_t port)
		{
			stop();
			TINY_GSM_YIELD();
			rx.clear();
			uint8_t newMux = -1;
			sock_connected = at->modemConnect(host, port, &newMux);
			if (sock_connected) {
				mux = newMux;
				at->sockets[mux] = this;
			}
			return sock_connected;
		}

		virtual int connect(IPAddress ip, uint16_t port)
		{
			String host;
			host.reserve(16);
			host += ip[0];
			host += ".";
			host += ip[1];
			host += ".";
			host += ip[2];
			host += ".";
			host += ip[3];
			return connect(host.c_str(), port);
		}

		virtual void stop()
		{
			TINY_GSM_YIELD();
			at->sendAT(GF("+CIPCLOSE="), mux);
			sock_connected = false;
			at->waitResponse();
			rx.clear();
		}

		virtual size_t write(const uint8_t *buf, size_t size)
		{
			TINY_GSM_YIELD();
			//at->maintain();
			return at->modemSend(buf, size, mux);
		}

		virtual size_t write(uint8_t c)
		{
			return write(&c, 1);
		}

		virtual int available()
		{
			TINY_GSM_YIELD();
			if (!rx.size() && sock_connected) {
				at->maintain();
			}
			return rx.size();
		}

		virtual int read(uint8_t *buf, size_t size)
		{
			TINY_GSM_YIELD();
			size_t cnt = 0;
			while (cnt < size) {
				size_t chunk = TinyGsmMin(size-cnt, rx.size());
				if (chunk > 0) {
					rx.get(buf, chunk);
					buf += chunk;
					cnt += chunk;
					continue;
				}
				// TODO: Read directly into user buffer?
				if (!rx.size() && sock_connected) {
					at->maintain();
					//break;
				}
			}
			return cnt;
		}

		virtual int read()
		{
			uint8_t c;
			if (read(&c, 1) == 1) {
				return c;
			}
			return -1;
		}

		virtual int peek()
		{
			return -1;    //TODO
		}
		virtual void flush()
		{
			at->stream.flush();
		}

		virtual uint8_t connected()
		{
			if (available()) {
				return true;
			}
			return sock_connected;
		}
		virtual operator bool()
		{
			return connected();
		}

		/*
		 * Extended API
		 */

		String remoteIP() TINY_GSM_ATTR_NOT_IMPLEMENTED;

	private:
		TinyGsm*      at;
		uint8_t       mux;
		bool          sock_connected;
		RxFifo        rx;
	};

public:

	explicit TinyGsm(Stream& stream)
		: stream(stream)
	{
		memset(sockets, 0, sizeof(sockets));
	}

	/*
	 * Basic functions
	 */
	bool begin()
	{
		return init();
	}

	bool init()
	{
		if (!testAT()) {
			return false;
		}
		sendAT(GF("&FZE0"));  // Factory + Reset + Echo Off
		if (waitResponse() != 1) {
			return false;
		}
		sendAT(GF("+CMEE=0"));
		waitResponse();

		sendAT(GF("+CMER=3,0,0,2"));
		waitResponse();

		getSimStatus();
		return true;
	}

	void setBaud(unsigned long baud)
	{
		sendAT(GF("+IPR="), baud);
	}

	bool testAT(unsigned long timeout = 10000L)
	{
		for (unsigned long start = millis(); millis() - start < timeout; ) {
			sendAT(GF(""));
			if (waitResponse(200) == 1) {
				delay(100);
				return true;
			}
			delay(100);
		}
		return false;
	}

	void maintain()
	{
		waitResponse(10, NULL, NULL);
	}

	bool factoryDefault()
	{
		sendAT(GF("&FZE0&W"));  // Factory + Reset + Echo Off + Write
		waitResponse();
		sendAT(GF("&W"));       // Write configuration
		return waitResponse() == 1;
	}

	String getModemInfo()
	{
		sendAT(GF("I"));
		String res;
		if (waitResponse(1000L, res) != 1) {
			return "";
		}
		res.replace(GSM_NL "OK" GSM_NL, "");
		res.replace(GSM_NL, " ");
		res.trim();
		return res;
	}

	bool hasSSL()
	{
		return false;
	}

	/*
	 * Power functions
	 */

	bool restart()
	{
		if (!testAT()) {
			return false;
		}
		sendAT(GF("+RST=1"));
		delay(3000);
		return init();
	}

	bool poweroff()
	{
		sendAT(GF("+CPOF"));
		return waitResponse() == 1;
	}

	bool radioOff() TINY_GSM_ATTR_NOT_IMPLEMENTED;

	bool sleepEnable(bool enable = true) TINY_GSM_ATTR_NOT_IMPLEMENTED;

	/*
	 * SIM card functions
	 */

	bool simUnlock(const char *pin)
	{
		sendAT(GF("+CPIN=\""), pin, GF("\""));
		return waitResponse() == 1;
	}

	String getSimCCID()
	{
		sendAT(GF("+CCID"));
		if (waitResponse(GF(GSM_NL "+SCID: SIM Card ID:")) != 1) {
			return "";
		}
		String res = stream.readStringUntil('\n');
		waitResponse();
		res.trim();
		return res;
	}

	String getIMEI()
	{
		sendAT(GF("+GSN"));
		if (waitResponse(GF(GSM_NL)) != 1) {
			return "";
		}
		String res = stream.readStringUntil('\n');
		waitResponse();
		res.trim();
		return res;
	}

	SimStatus getSimStatus(unsigned long timeout = 10000L)
	{
		for (unsigned long start = millis(); millis() - start < timeout; ) {
			sendAT(GF("+CPIN?"));
			if (waitResponse(GF(GSM_NL "+CPIN:")) != 1) {
				delay(1000);
				continue;
			}
			int status = waitResponse(GF("READY"), GF("SIM PIN"), GF("SIM PUK"));
			waitResponse();
			switch (status) {
			case 2:
			case 3:
				return SIM_LOCKED;
			case 1:
				return SIM_READY;
			default:
				return SIM_ERROR;
			}
		}
		return SIM_ERROR;
	}

	RegStatus getRegistrationStatus()
	{
		sendAT(GF("+CREG?"));
		if (waitResponse(GF(GSM_NL "+CREG:")) != 1) {
			return REG_UNKNOWN;
		}
		streamSkipUntil(','); // Skip format (0)
		int status = stream.readStringUntil('\n').toInt();
		waitResponse();
		return (RegStatus)status;
	}

	String getOperator()
	{
		sendAT(GF("+COPS=3,0")); // Set format
		waitResponse();

		sendAT(GF("+COPS?"));
		if (waitResponse(GF(GSM_NL "+COPS:")) != 1) {
			return "";
		}
		streamSkipUntil('"'); // Skip mode and format
		String res = stream.readStringUntil('"');
		waitResponse();
		return res;
	}

	/*
	 * Generic network functions
	 */

	int getSignalQuality()
	{
		sendAT(GF("+CSQ"));
		if (waitResponse(GF(GSM_NL "+CSQ:")) != 1) {
			return 99;
		}
		int res = stream.readStringUntil(',').toInt();
		waitResponse();
		return res;
	}

	bool isNetworkConnected()
	{
		RegStatus s = getRegistrationStatus();
		return (s == REG_OK_HOME || s == REG_OK_ROAMING);
	}

	bool waitForNetwork(unsigned long timeout = 60000L)
	{
		for (unsigned long start = millis(); millis() - start < timeout; ) {
			if (isNetworkConnected()) {
				return true;
			}
			delay(250);
		}
		return false;
	}

	/*
	 * GPRS functions
	 */
	bool gprsConnect(const char* apn, const char* user = NULL, const char* pwd = NULL)
	{
		gprsDisconnect();

		sendAT(GF("+CGATT=1"));
		if (waitResponse(60000L) != 1) {
			return false;
		}

		// TODO: wait AT+CGATT?

		sendAT(GF("+CGDCONT=1,\"IP\",\""), apn, '"');
		waitResponse();

		if (!user) {
			user = "";
		}
		if (!pwd) {
			pwd = "";
		}
		sendAT(GF("+CSTT=\""), apn, GF("\",\""), user, GF("\",\""), pwd, GF("\""));
		if (waitResponse(60000L) != 1) {
			return false;
		}

		sendAT(GF("+CGACT=1,1"));
		waitResponse(60000L);

		sendAT(GF("+CIPMUX=1"));
		if (waitResponse() != 1) {
			return false;
		}

		return true;
	}

	bool gprsDisconnect()
	{
		// Shut the TCP/IP connection
		sendAT(GF("+CIPSHUT"));
		if (waitResponse(60000L) != 1) {
			return false;
		}

		for (int i = 0; i<3; i++) {
			sendAT(GF("+CGATT=0"));
			if (waitResponse(5000L) == 1) {
				return true;
			}
		}

		return false;
	}

	bool isGprsConnected()
	{
		sendAT(GF("+CGATT?"));
		if (waitResponse(GF(GSM_NL "+CGATT:")) != 1) {
			return false;
		}
		int res = stream.readStringUntil('\n').toInt();
		waitResponse();
		return (res == 1);
	}

	String getLocalIP()
	{
		sendAT(GF("+CIFSR"));
		String res;
		if (waitResponse(10000L, res) != 1) {
			return "";
		}
		res.replace(GSM_NL "OK" GSM_NL, "");
		res.replace(GSM_NL, "");
		res.trim();
		return res;
	}

	IPAddress localIP()
	{
		return TinyGsmIpFromString(getLocalIP());
	}

	/*
	 * Phone Call functions
	 */

	bool setGsmBusy(bool busy = true) TINY_GSM_ATTR_NOT_AVAILABLE;

	bool callAnswer()
	{
		sendAT(GF("A"));
		return waitResponse() == 1;
	}

	// Returns true on pick-up, false on error/busy
	bool callNumber(const String& number)
	{
		if (number == GF("last")) {
			sendAT(GF("DLST"));
		} else {
			sendAT(GF("D\""), number, "\";");
		}

		if (waitResponse(5000L) != 1) {
			return false;
		}

		if (waitResponse(60000L,
		                 GF(GSM_NL "+CIEV: \"CALL\",1"),
		                 GF(GSM_NL "+CIEV: \"CALL\",0"),
		                 GFP(GSM_ERROR)) != 1) {
			return false;
		}

		int rsp = waitResponse(60000L,
		                       GF(GSM_NL "+CIEV: \"SOUNDER\",0"),
		                       GF(GSM_NL "+CIEV: \"CALL\",0"));

		int rsp2 = waitResponse(300L, GF(GSM_NL "BUSY" GSM_NL), GF(GSM_NL "NO ANSWER" GSM_NL));

		return rsp == 1 && rsp2 == 0;
	}

	bool callHangup()
	{
		sendAT(GF("H"));
		return waitResponse() == 1;
	}

	// 0-9,*,#,A,B,C,D
	bool dtmfSend(char cmd, unsigned duration_ms = 100)
	{
		duration_ms = constrain(duration_ms, 100, 1000);

		// The duration parameter is not working, so we simulate it using delay..
		// TODO: Maybe there's another way...

		//sendAT(GF("+VTD="), duration_ms / 100);
		//waitResponse();

		sendAT(GF("+VTS="), cmd);
		if (waitResponse(10000L) == 1) {
			delay(duration_ms);
			return true;
		}
		return false;
	}

	/*
	 * Audio functions
	 */

	bool audioSetHeadphones()
	{
		sendAT(GF("+SNFS=0"));
		return waitResponse() == 1;
	}

	bool audioSetSpeaker()
	{
		sendAT(GF("+SNFS=1"));
		return waitResponse() == 1;
	}

	bool audioMuteMic(bool mute)
	{
		sendAT(GF("+CMUT="), mute);
		return waitResponse() == 1;
	}

	/*
	 * Messaging functions
	 */

	String sendUSSD(const String& code)
	{
		sendAT(GF("+CMGF=1"));
		waitResponse();
		sendAT(GF("+CSCS=\"HEX\""));
		waitResponse();
		sendAT(GF("+CUSD=1,\""), code, GF("\",15"));
		if (waitResponse(10000L) != 1) {
			return "";
		}
		if (waitResponse(GF(GSM_NL "+CUSD:")) != 1) {
			return "";
		}
		stream.readStringUntil('"');
		String hex = stream.readStringUntil('"');
		stream.readStringUntil(',');
		int dcs = stream.readStringUntil('\n').toInt();

		if (dcs == 15) {
			return TinyGsmDecodeHex7bit(hex);
		} else if (dcs == 72) {
			return TinyGsmDecodeHex16bit(hex);
		} else {
			return hex;
		}
	}

	bool sendSMS(const String& number, const String& text)
	{
		sendAT(GF("+CMGF=1"));
		waitResponse();
		sendAT(GF("+CMGS=\""), number, GF("\""));
		if (waitResponse(GF(">")) != 1) {
			return false;
		}
		stream.print(text);
		stream.write((char)0x1A);
		stream.flush();
		return waitResponse(60000L) == 1;
	}


	/*
	 * Location functions
	 */

	String getGsmLocation() TINY_GSM_ATTR_NOT_AVAILABLE;

	/*
	 * Battery functions
	 */

	uint16_t getBattVoltage() TINY_GSM_ATTR_NOT_AVAILABLE;

	int getBattPercent()
	{
		sendAT(GF("+CBC?"));
		if (waitResponse(GF(GSM_NL "+CBC:")) != 1) {
			return false;
		}
		stream.readStringUntil(',');
		int res = stream.readStringUntil('\n').toInt();
		waitResponse();
		return res;
	}

protected:

	bool modemConnect(const char* host, uint16_t port, uint8_t* mux)
	{
		sendAT(GF("+CIPSTART="),  GF("\"TCP"), GF("\",\""), host, GF("\","), port);

		if (waitResponse(75000L, GF(GSM_NL "+CIPNUM:")) != 1) {
			return false;
		}
		int newMux = stream.readStringUntil('\n').toInt();

		int rsp = waitResponse(75000L,
		                       GF("CONNECT OK" GSM_NL),
		                       GF("CONNECT FAIL" GSM_NL),
		                       GF("ALREADY CONNECT" GSM_NL));
		if (waitResponse() != 1) {
			return false;
		}
		*mux = newMux;

		return (1 == rsp);
	}

	int modemSend(const void* buff, size_t len, uint8_t mux)
	{
		sendAT(GF("+CIPSEND="), mux, ',', len);
		if (waitResponse(2000L, GF(GSM_NL ">")) != 1) {
			return 0;
		}
		stream.write((uint8_t*)buff, len);
		stream.flush();
		if (waitResponse(10000L, GFP(GSM_OK), GF(GSM_NL "FAIL")) != 1) {
			return 0;
		}
		return len;
	}

	bool modemGetConnected(uint8_t mux)
	{
		sendAT(GF("+CIPSTATUS")); //TODO mux?
		int res = waitResponse(GF(",\"CONNECTED\""), GF(",\"CLOSED\""), GF(",\"CLOSING\""),
		                       GF(",\"INITIAL\""));
		waitResponse();
		return 1 == res;
	}

public:

	/* Utilities */

	template<typename T>
	void streamWrite(T last)
	{
		stream.print(last);
	}

	template<typename T, typename... Args>
	void streamWrite(T head, Args... tail)
	{
		stream.print(head);
		streamWrite(tail...);
	}

	bool streamSkipUntil(char c)   //TODO: timeout
	{
		while (true) {
			while (!stream.available()) {
				TINY_GSM_YIELD();
			}
			if (stream.read() == c) {
				return true;
			}
		}
		return false;
	}

	template<typename... Args>
	void sendAT(Args... cmd)
	{
		streamWrite("AT", cmd..., GSM_NL);
		stream.flush();
		TINY_GSM_YIELD();
		//DBG("### AT:", cmd...);
	}

	// TODO: Optimize this!
	uint8_t waitResponse(uint32_t timeout, String& data,
	                     GsmConstStr r1=GFP(GSM_OK), GsmConstStr r2=GFP(GSM_ERROR),
	                     GsmConstStr r3=NULL, GsmConstStr r4=NULL, GsmConstStr r5=NULL)
	{
		/*String r1s(r1); r1s.trim();
		String r2s(r2); r2s.trim();
		String r3s(r3); r3s.trim();
		String r4s(r4); r4s.trim();
		String r5s(r5); r5s.trim();
		DBG("### ..:", r1s, ",", r2s, ",", r3s, ",", r4s, ",", r5s);*/
		data.reserve(64);
		int index = 0;
		unsigned long startMillis = millis();
		do {
			TINY_GSM_YIELD();
			while (stream.available() > 0) {
				int a = stream.read();
				if (a <= 0) {
					continue;    // Skip 0x00 bytes, just in case
				}
				data += (char)a;
				if (r1 && data.endsWith(r1)) {
					index = 1;
					goto finish;
				} else if (r2 && data.endsWith(r2)) {
					index = 2;
					goto finish;
				} else if (r3 && data.endsWith(r3)) {
					index = 3;
					goto finish;
				} else if (r4 && data.endsWith(r4)) {
					index = 4;
					goto finish;
				} else if (r5 && data.endsWith(r5)) {
					index = 5;
					goto finish;
				} else if (data.endsWith(GF("+CIPRCV:"))) {
					int mux = stream.readStringUntil(',').toInt();
					int len = stream.readStringUntil(',').toInt();
					int len_orig = len;
					if (len > sockets[mux]->rx.free()) {
						DBG("### Buffer overflow: ", len, "->", sockets[mux]->rx.free());
					} else {
						DBG("### Got: ", len, "->", sockets[mux]->rx.free());
					}
					while (len--) {
						while (!stream.available()) {
							TINY_GSM_YIELD();
						}
						sockets[mux]->rx.put(stream.read());
					}
					if (len_orig > sockets[mux]->available()) { // TODO
						DBG("### Fewer characters received than expected: ", sockets[mux]->available(), " vs ", len_orig);
					}
					data = "";
				} else if (data.endsWith(GF("+TCPCLOSED:"))) {
					int mux = stream.readStringUntil('\n').toInt();
					if (mux >= 0 && mux < TINY_GSM_MUX_COUNT) {
						sockets[mux]->sock_connected = false;
					}
					data = "";
					DBG("### Closed: ", mux);
				}
			}
		} while (millis() - startMillis < timeout);
finish:
		if (!index) {
			data.trim();
			if (data.length()) {
				DBG("### Unhandled:", data);
			}
			data = "";
		}
		return index;
	}

	uint8_t waitResponse(uint32_t timeout,
	                     GsmConstStr r1=GFP(GSM_OK), GsmConstStr r2=GFP(GSM_ERROR),
	                     GsmConstStr r3=NULL, GsmConstStr r4=NULL, GsmConstStr r5=NULL)
	{
		String data;
		return waitResponse(timeout, data, r1, r2, r3, r4, r5);
	}

	uint8_t waitResponse(GsmConstStr r1=GFP(GSM_OK), GsmConstStr r2=GFP(GSM_ERROR),
	                     GsmConstStr r3=NULL, GsmConstStr r4=NULL, GsmConstStr r5=NULL)
	{
		return waitResponse(1000, r1, r2, r3, r4, r5);
	}

public:
	Stream&       stream;

protected:
	GsmClient*    sockets[TINY_GSM_MUX_COUNT];
};

#endif
