import threading
from time import time, sleep
from lib.utils import Utils
from lib.minimalmodbus import Instrument, MODE_RTU
import serial
from struct import pack, unpack

class FlowLib(Utils):
    def __init__(self, usbdevnode, modbus_id, baudrate: int = 9600, log_id = "FLOW") -> None:
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.port = usbdevnode.get_devnode()
        self.modbus_id = modbus_id
        self.last_flow_timestamp = time()
        self.instrument = []
        self.connect()
        threading.Thread(target = self.read).start()

    def connect(self) -> None:
        try:
            for i in range(2):
                client = Instrument(self.port, self.modbus_id[i], close_port_after_each_call=True, debug=False)
                client.serial.baudrate = 9600
                self.instrument.append(client)
        except:
            self.traceback()
            sleep(1)

    def read_float_reg(self,regOne, regTwo, instrument):
        data = (instrument.read_register(
            regOne), instrument.read_register(regTwo))
        packed_string = pack("HH", *data)
        unpacked_string = unpack("f", packed_string)[0]
        return float("{:.2f}".format(unpacked_string))

    def read_flow(self,instrument):
        # Convert m3/h to l/s
        flow = self.read_float_reg(0, 1, instrument) * 1000 / 3600
        print(f'Flow rate: {flow} l/s')
        return flow

    def read_signal_quality(self, instrument):
        return(instrument.read_register(91)& 0x00FF)


    def read_flowmeter(self, instrument):
        flujo, quality = -1, -1
        try:
            ## Read 
            flujo = self.read_flow(instrument)
            quality = self.read_signal_quality(instrument)

            self.log(f"Flujo = {flujo} y Quality={quality}")
            self.last_flow_timestamp = time()
            return flujo, quality
        except:
            self.log(f"Error reading flowmeter modbus id: {instrument.address}")
        
        return flujo, quality
    
    def read(self) -> None:
        while True:
            try:
                # Read Left Flowmeter
                flujo_izq, q_izq = self.read_flowmeter(self.instrument[0])
                sleep(0.4)
                # Read Right Flowmeter
                flujo_der, q_der = self.read_flowmeter(self.instrument[1])
                sleep(0.4)
                self.process_line(flujo_izq, flujo_der, q_izq, q_der)
            except:
                self.traceback()
                sleep(1)
            

    def process_line(self, flow_left, flow_right, sq_left, sq_right) -> None:
        try:
            data_dict = {
                "f_l": flow_left,
                "sq_l": sq_left,
                "f_r": flow_right,
                "sq_r": sq_right,
                "timestamp": int(time()*1000)
            }
            self.log(f"Flow Data: {data_dict}")
            self.emit(data_type=self.log_id, data=data_dict)

        except:
            self.traceback() 
               


    

