import io
import pynmea2
import threading
import threading
from serial import Serial, SerialException, serialutil
from time import time, sleep
from lib.utils import Utils
import json


class GpsLib(Utils):

	def __init__(self, usbdevnode, baudrate = 9600, log_id="GPS_USB"):
		self.baudrate = baudrate
		self.log_id = log_id
		self.usbdevnode = usbdevnode
		self.serial_module = None
		self.seconds_measure = 30
		self.last_gps_timestamp = time()
		self.timer = serialutil.Timeout(2)
		threading.Thread(target = self.read).start()



	def connect(self) -> None:
		"""
		This function attempts to establish a serial connection with the specified USB device node.

		"""

		try:
			self.log(f"Try to connect serial port: {self.usbdevnode.get_devnode()}")
			self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=1)
			self.sio = io.TextIOWrapper(io.BufferedRWPair(self.serial_module, self.serial_module))



		except Exception as Ex:
			self.log(Ex)
	


	def read(self) -> None:
		"""
		This function continuously reads lines from the serial module and processes them.
		If the line is empty, the function skips it.

		"""

		self.log("Reading line from serial")
		while True:
			# Read every 5 minutes
			if self.timer.expired():
				if self.serial_module is None:
					self.connect()
				
				else:
					line = self.sio.readline()
					if line:
						if self.process_line(line): self.timer.restart(self.seconds_measure)
			sleep(0.1)

	def process_line(self, data):
		try:			
			data = data.replace("\n", "")
			if "RMC" in data:
				self.log("GOT [%s]" % data)
				result = self.decode_rmc(data)
				if result is not None:
					dict_data = {
						'latitude': result[0],
						'longitude': result[1],
						'speed': result[2],
						'timestamp': int(time()),
					}
					self.log("GPS: [%s]" % dict_data)
					self.emit(data_type=self.log_id, data= dict_data)
					self.last_gps_timestamp = time()
					self.write_file("/gps_live.json", json.dumps(dict_data), "w")
					return True

		except Exception as ex:
			self.traceback()
		return False
	
	def decode_rmc(self, line):

		try:
			self.text_gprmc = line
			line = pynmea2.parse(line)

			# Get current position
			lat = round(line.latitude, 6)
			lon = round(line.longitude, 6)
			speed = round(line.spd_over_grnd, 6)
			return lat, lon, speed

		except ValueError as exc:
			self.log("GPS NOFIX: Unable to find Lat/Lon in NMEA frame [%s]" % line)
		except Exception as ex:
			self.traceback()
			return None




