import threading
from serial import Serial
from time import time, sleep
from lib.utils import Utils
from gps.gps_lib import GpsLib


class SerialLib(Utils):
    def __init__(self, usbdevnode, config, baudrate: int = 115200, log_id = "SERIAL") -> None:
        self.config = config
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.usbdevnode = usbdevnode
        self.gps_model = config["GPS_SENSOR"]["MODEL"]
        self.limits_pm_value = [config["DUST_SENSOR"]["MIN_VALUE_PM100"], config["DUST_SENSOR"]["MAX_VALUE_PM100"]]
        self.config_evo = config["DUST_SENSOR"]["EVOS"]
        self.last_dust_timestamp = time()
        if self.gps_model !="GPS_USB":
            MIN_VALUE_SPEED = config["GPS_SENSOR"]["MIN_VALUE_SPEED"]
            self.gps = GpsLib(usbdevnode= usbdevnode, model= self.gps_model, min_value_speed= MIN_VALUE_SPEED, log_id= "GPS_MASTER")
            self.gps.set_server(config["SERVER"]["IP"], config["SERVER"]["PORT"])
            self.log("GPS MASTER initialized")
        threading.Thread(target = self.connect).start()

    def connect(self) -> None:
        """
        This function attempts to establish a serial connection with the specified USB device node.

        """

        try:
            self.log(f"Try to connect serial port: {self.usbdevnode.get_devnode()}")
            self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=self.timeout)
            self.read()

        except Exception as Ex:
            self.log(Ex)

    def read(self) -> None:
        """
        This function continuously reads lines from the serial module and processes them.
        If the line is empty, the function skips it.

        """

        self.log("Reading line from serial")
        while True:
            try:
                try:
                    raw_line = self.serial_module.readline().decode("utf-8")
                except: pass
                line = raw_line.strip().replace("\n", "")
                if line =="":
                    pass
                else:
                    self.log(f"GOT: {line.lower()}")
                    self.process_line(line)
            except:
                self.traceback()
            sleep(0.01)

    def process_line(self, line: str) -> None:
        """
        This function processes the received line from the serial module.
        It checks the type of the line and calls the corresponding processing function.

        Parameters:
        line (str): The received line from the serial module.

        """

        if 'pm100' in line:
            payload = line.lower().split('[', 1)[1].split(']')[0].split("/")
            self.process_dust(payload)

        elif 'RMC' in line:
            payload = line.split("-->")
            self.process_gps(payload[1])

        elif 'TEMPERATURE' in line:
            payload = line.split("-->")
            self.process_temperature(payload)

        elif 'RX' in line:
            pass

        else:
            print(f"That's not a valid type {line}")

    def process_dust(self, payload: list) -> None:
        """
        Processes the dust sensor data received from the NRF module.

        Parameters:
        payload (list): A list containing the raw data from the dust sensor. The first element is the EVO model,
                        and the third element is the PM100 value.

        Returns:
        None: This function does not return any value. It processes the dust sensor data and emits it if valid.
        """
        evo = payload[0]
        if self.config_evo == [] or evo in self.config_evo:
            #  Multiply by 10
            pm100_value = int(payload[2])*10
            pm100, valid = self.filter_data(pm100_value) 
            if valid:
                self.log(f"Valid pm100: {pm100}")
                data_dict = {
                    "pm100": pm100,
                    "timestamp": int(time()*1000)
                }
                self.emit(data_type=self.log_id, data=data_dict)
        else:
            self.log("EVO NO CORRESPONDE AL CONFIGURADO")
        self.last_dust_timestamp = time()

    def process_gps(self, payload: str) -> None:
        """
        Processes the GPS data received from serial module.

        Parameters:
        payload (str): A string containing the raw GPS data received from serial module.

        Returns:
        None: This function does not return any value. It processes the GPS data and passes if the GPS model is not "GPS_USB".
        """

        if self.gps_model != "GPS_USB":
            self.gps.process_line(payload)
 


    def process_temperature(self, payload: list) -> None:
        """
        Processes the temperature data received from the serial module.

        Parameters:
        payload (list): A list containing the raw temperature data received from the serial module.
                        The first element is the raw temperature value, and the second element is the unit.

        Returns:
        None: This function does not return any value. It processes the temperature data and emits it.
        """
        data_dict = {
            "temperature": float(payload[1].split()[0]),
            "timestamp": int(time())
        }
        self.emit(data_type="TEMPERATURE", data=data_dict)

    def filter_data(self, value: float) -> tuple:
        """
        Filters the PM100 dust sensor data based on the specified limits.

        Parameters:
        value (float): The raw PM100 dust sensor value to be filtered.

        Returns:
        tuple: A tuple containing the filtered PM100 value and a boolean indicating whether the value is valid.
               If the value is within the specified limits, the tuple is (value, True).
               If the value is outside the specified limits, the tuple is (None, False).
        """
        if self.limits_pm_value[0] < value <= self.limits_pm_value[1]:
                return value, True
        else: print(f"valor {value} esta fuera de los limites")

        return None, False

    



