from datetime import datetime
from time import sleep
import traceback
import requests
import json
import sys

CONFIG_SOURCE = "/srv/datalogger_ecom/config_ecom.json"
sys.path.append('/srv/datalogger_ecom/')
from lib import mqtt
from lib.utils import Utils
from database.models import Database

class Upload(Utils):
    def __init__(self, config_path):
        self.log_id = "UPLOAD"
        self.config_path= config_path
        self.machine_id = self.get_product_id()
        self.machine_name = self.get_product_name()
        self.faena = self.get_faena_assigned()
        self.database = Database()
        self.update_topic()

    def update_topic(self):
        try:
            f = open(self.config_path)
            config:dict = json.load(f)

            # MINING
            self.activated_mining = config["AUTOUPLOAD"]["MINING"]["ACTIVATED"] 
            self.topic = config["AUTOUPLOAD"]["MINING"]["DATA"]

            # EYE3
            self.activated_eye3 = config["AUTOUPLOAD"]["EYE3"]["ACTIVATED"] 
            self.url = config["AUTOUPLOAD"]["EYE3"]["DATA"]

        except:
            self.activated_mining = 0
            self.topic = ""

            self.activated_eye3 = 0
            self.url = ""

            self.traceback()

    def check_data(self):
        try:
            print(f"Check data {self.get_datetime()}")
            # 1. Check data MINING
            if self.activated_mining:
                print("CHECK DATA MINING")
                data, _ = self.database.get_ecom_data(condition_column= "uploaded_mining")
                if data and self.topic:
                    # 1a. Send data to Mining MQTT
                    ids_success = self.upload_mqtt(data, self.topic)
                    # 1b. Update column uploaded_mining with value 1
                    if ids_success: self.database.update_value(table="ecom_data", column_name= "uploaded_mining", ids=ids_success)
                else:
                    print("No data to upload to Mining")
            else:
                print("Envio de data a MINING esta desactivado")
             
            # 2. Check data EYE3
            if self.activated_eye3:
                print("CHECK DATA EYE3")
                data, _ = self.database.get_ecom_data(condition_column= "uploaded_eye3")
                if data and self.url:
                    # 2a. Send data to EYE3 HTTP
                    ids_success = self.upload_http(data, self.url)
                    # 2b. Update column uploaded_mining with value 1
                    if ids_success: self.database.update_value(table="ecom_data", column_name= "uploaded_eye3", ids=ids_success)

                else:
                    print("No data to upload to EYE3")
            else:
                print("Envio de data a EYE3 esta desactivado")
        
        except:
            self.traceback()

    def upload_mqtt(self, data, topic):
        print(f"Try to upload data to {topic}")
        ids = []
        client = mqtt.connect()
        try:
            if client.is_connected():
                sub_topic = f"acks/{topic}" 
                client.subscribe(sub_topic)        
                for row in data:
                    packet = []
                    row_data = row[1].replace('nan', '0')
                    for microdata in eval(row_data):
                        dict_microdata = {
                                "measurement": {
                                    "latitude": microdata["latitude"],
                                    "longitude": microdata["longitude"],
                                    "speed_kmh": microdata["speed"],
                                    "pm100": microdata["pm100"],
                                    "geofence_id": microdata["zone_id"]
                                },
                                "time": microdata["timestamp"]
                            }
                        packet.append(dict_microdata)

                    dict_data = {
                        "packet": packet,
                        "id": row[0],
                        "name": self.machine_name,
                        "machineid": int(self.machine_id)
                    }
                    print(f"Data to upload: {dict_data}")

                    result = mqtt.publish(client= client, topic=topic, dict_data=dict_data)
                    if result:
                        print(f"Data uploaded Success to {topic}")
                        res = mqtt.get_response(wait=5)
                        if res:
                            if row[0] == res.get("id", -1) and res.get("status", "error") == "success": ids.append(row[0])
                            else: print("Error al escribir en la base de datos del server")
                        else: print("No hay respuesta del server")

                client.loop_stop()
                client.disconnect()
                return ids
            else:
                print("No se logró conectar a MQTT")
        except:
            self.traceback()
        return ids


    def upload_http(self, data, url):
        ids = []
        for row in data:
            puntos = []
            row_data= row[1].replace('nan','0')
            for microdata in eval(row_data):
                json_data ={
                    "truck": str(self.machine_id),
                    "lat": round(microdata["latitude"],6),
                    "lon": round(microdata["longitude"],6),
                    "speed": microdata["speed"],
                    "dist": int(microdata["distance"]),
                    "pm10pcc":int(microdata["pm100"]/10), # divide by 10 pm100 to eye3
                    "pm25pcc": 0,
                    "date": datetime.fromtimestamp(int(microdata["timestamp"]/1000)).strftime('%Y-%m-%d %H:%M:%S'),
                }
                print(json_data)
                puntos.append(json_data)

            json_item = {
                    "idpoligono": 0,
                    "tasaideal": 0.0,
                    "horaingreso": puntos[0]["date"],
                    "horasalida": puntos[-1]["date"],
                    "pm10": float(row[2]/10), # divide by 10 pm100 to eye3
                    "pm25": 0.0,
                    "velocidadpromedio": float(row[5]),
                    "vehiculo": self.machine_id,
                    "puntos": puntos
                }
            try:
                if self.online():
                    print("Uploading item to %s", url)
                    r = requests.post(url= url, data = {'json': json.dumps(json_item)}, timeout=5)
                    content = r.content
                    if '''{"ok":true}''' in content.decode("utf-8"):
                        ids.append(row[0])
                        print("Item number uploaded successfully to %s", url)
                    else:
                        print("Upload FAILED. Will retry in 35 seconds")
                        sleep(1)

            except requests.exceptions.ConnectionError:
                print("Failed to upload to %s [ConnectionError]" % (url))

            except Exception as err:
                print("Failed to upload to %s [unknown error]" % (url))
                print(repr(err))
            sleep(0.5)
        return ids


if __name__ == "__main__":
    print("---")
    print("INIT AUTOUPLOAD")
    upload = Upload(CONFIG_SOURCE) 
    while True:
        try:
            upload.check_data()
        except:
            print("error")
            e = sys.exc_info()
            print("dumping traceback for [%s: %s]" % (str(e[0].__name__), str(e[1])))
            traceback.print_tb(e[2])
            foo = "bar" # do nothing
        sleep(10)
        