import json
from time import sleep
import traceback
import sqlite3
import sys
from datetime import datetime, timedelta
import requests

CONFIG_SOURCE = "/srv/datalogger_gimp/config_gimp.json"
DATABASE_PATH = "/srv/datalogger_gimp/database/database.db"

sys.path.append('/srv/datalogger_gimp/')
from lib.utils import Utils
from lib import mqtt
from database.models import Database

class Upload(Utils):
    def __init__(self, config_path, database_path):
        self.log_id = "UPLOAD"
        self.config_path= config_path
        self.database_path = database_path
        self.machine_id = self.get_product_id()
        self.machine_name = self.get_product_name()
        self.faena = self.get_faena_assigned()
        self.avalaible_faenas = self.get_avalaible_faenas(datalogger_type= "gimp")
        self.database = Database()
        self.update_topic()

    def update_topic(self):
        try:
            f = open(self.config_path)
            config:dict = json.load(f)

            # MINING
            self.activated_mining = config["AUTOUPLOAD"]["MINING"]["ACTIVATED"] 
            self.topic = config["AUTOUPLOAD"]["MINING"]["DATA"]

            # EYE3
            self.activated_eye3 = config["AUTOUPLOAD"]["EYE3"]["ACTIVATED"] 
            self.url = config["AUTOUPLOAD"]["EYE3"]["DATA"]

        except:
            self.activated_mining = 0
            self.topic = ""

            self.activated_eye3 = 0
            self.url = ""

            self.traceback()

    def check_data(self):
        try:
            if self.online:
                print(f"Check data {self.get_datetime()}")
                # 1. Check data MINING
                if self.activated_mining:
                    print("CHECK DATA MINING")
                    data, _ = self.database.get_gimp_data(condition_column= "uploaded_mining")
                    if data and self.topic:
                        # 1a. Send data to Mining MQTT
                        ids_success = self.upload_mqtt(data, self.topic)
                        # 1b. Update column uploaded_mining with value 1
                        if ids_success: self.database.update_value(table="gimp_data", column_name= "uploaded_mining", ids=ids_success)
                    else:
                        print("No data to upload to Mining")
                    
                # 2. Check data EYE3
                if self.activated_eye3:
                    print("CHECK DATA EYE3")
                    data, _ = self.database.get_gimp_data(condition_column= "uploaded_eye3")
                    if data:
                        if self.faena not in self.avalaible_faenas:
                            print("We aren't in faena, set uploaded_eye3 1 this data")
                            ids = [row[0] for row in data]
                            self.database.update_value(table="gimp_data", column_name= "uploaded_eye3", ids=ids)
                        else:
                            if self.url:
                                # 2a. Send data to EYE3 HTTP
                                ids_success = self.upload_http(data, self.url)
                                # 2b. Update column uploaded_mining with value 1
                                if ids_success: self.database.update_value(table="gimp_data", column_name= "uploaded_eye3", ids=ids_success)
                    else:
                        print("No data to upload to EYE3")
            else: print("There is no internet connection")

        except:
            self.traceback()

    def upload_mqtt(self, data, topic):
        print(f"Try to upload data to {topic}")
        ids = []
        client = mqtt.connect()
        try:
            if client.is_connected():
                sub_topic = f"acks/{topic}" 
                client.subscribe(sub_topic)               
                for row in data:
                    dict_data = {
                        "measurement": {
                            "machineid": self.machine_id,
                            "pm10_ugm3": row[2] if row[2]!= None else -1,
                            "pm100_ugm3": row[1] if row[1]!= None else -1,
                            },
                        "equipo": {
                            "name": self.machine_name
                            },
                        "time": row[14]*1000,
                        "id": row[0]
                        }
                    
                    
                    print(f"Data to upload: {dict_data}")
                    result = mqtt.publish(client= client, topic=topic, dict_data=dict_data)
                    if result:
                        print(f"Data uploaded Success to {topic}")
                        res = mqtt.get_response(wait=2)
                        if res:
                            if row[0] == res.get("id", -1) and res.get("status", "error") == "success": ids.append(row[0])
                            else: print("Error al escribir en la base de datos del server")
                        else: print("No hay respuesta del server")
                                

                client.loop_stop()
                client.disconnect()
                return ids
                
            else:
                print("No se logró conectar a MQTT")
        except:
            self.traceback()
        return ids



    def upload_http(self, data, url):
        print(f"Try to upload data to {url}")
        ids = []
        for row in data:
            # Convert data to eye3 format
            packet_data ={
                        'station': self.machine_id,
                        'rain': row[10],
                        'ws': row[5],
                        'wd': row[6],
                        'temp': row[8],
                        'hum': row[9],
                        'press': 0,
                        'boxtemp': 0,
                        'pm10': row[2],
                        'pm25': 0,
                        'pm100': row[1]/10 if row[1]!= None else None, 
                        'pm10bam': 0,
                        'lat': row[3],
                        'lon': row[4],
                        #'date': row[13],
                        'solarrad': row[12],
                        'uvindex': row[11]
                }
            # Minus 15 minutes to datetime
            packet_data["date"] = (datetime.strptime(row[13], '%Y-%m-%d %H:%M:%S') - timedelta(minutes=15)).strftime('%Y-%m-%d %H:%M:%S')

            try:
                print(f"Data to upload: {packet_data}")
                r = requests.post(url= url, data = {'json': json.dumps([packet_data])}, timeout=2)
                content = r.content.decode("utf-8")
                print(content)
                if "Insert OK!" in content or "Ya existe" in content:
                    ids.append(row[0])
                    print("Item number uploaded successfully to %s", url)
                else:
                    print("Upload FAILED. Will retry in  seconds")
                    sleep(1)

            except requests.exceptions.ConnectionError:
                print("Failed to upload to %s [ConnectionError]" % (url))

            except Exception as err:
                print("Failed to upload to %s [unknown error]" % (url))
                print(repr(err))
            sleep(0.5)
        return ids


if __name__ == "__main__":
    print("---")
    print("INIT AUTOUPLOAD")
    upload = Upload(CONFIG_SOURCE, DATABASE_PATH) 
    while True:
        try:
            upload.check_data()
        except:
            print("error")
            e = sys.exc_info()
            print("dumping traceback for [%s: %s]" % (str(e[0].__name__), str(e[1])))
            traceback.print_tb(e[2])
            foo = "bar" # do nothing
        sleep(60)
        
