from datetime import datetime
from time import sleep
import traceback
import requests
import json
import sys

CONFIG_SOURCE = "/srv/datalogger_mmr/config_mmr.json"
sys.path.append('/srv/datalogger_mmr/')
from lib import mqtt
from lib.utils import Utils
from database.models import Database

class Upload(Utils):
    def __init__(self, config_path):
        self.log_id = "UPLOAD"
        self.config_path= config_path
        self.machine_id = self.get_product_id()
        self.machine_name = self.get_product_name()
        self.faena = self.get_faena_assigned()
        self.avalaible_faenas = self.get_avalaible_faenas(datalogger_type= "mmr")
        self.database = Database()
        self.update_topic()

    def update_topic(self):
        try:
            f = open(self.config_path)
            config:dict = json.load(f)

            # MINING
            self.activated_mining = config["AUTOUPLOAD"]["MINING"]["ACTIVATED"] 
            self.topic = config["AUTOUPLOAD"]["MINING"]["DATA"]

            # EYE3
            self.activated_eye3 = config["AUTOUPLOAD"]["EYE3"]["ACTIVATED"] 
            self.url = config["AUTOUPLOAD"]["EYE3"]["DATA"]

        except:
            self.activated_mining = 0
            self.topic = ""

            self.activated_eye3 = 0
            self.url = ""

            self.traceback()

    def check_data(self):
        try:
            print(f"Check data {self.get_datetime()}")
            # 1. Check data MINING
            if self.activated_mining:
                print("CHECK DATA MINING")
                data, _ = self.database.get_mmr_data(condition_column= "uploaded_mining")
                if data and self.topic:
                    # 1a. Send data to Mining MQTT
                    ids_success = self.upload_mqtt(data, self.topic)
                    # 1b. Update column uploaded_mining with value 1
                    if ids_success: self.database.update_value(table="mmr_data", column_name= "uploaded_mining", ids=ids_success)
                else:
                    print("No data to upload to Mining")
            else:
                print("Envio de data a MINING esta desactivado")
             
            # 2. Check data EYE3
            if self.activated_eye3:
                print("CHECK DATA EYE3")
                data, _ = self.database.get_mmr_data(condition_column= "uploaded_eye3")
                if data:
                    if self.faena not in self.avalaible_faenas:
                        print("We aren't in faena, set uploaded_eye3 1 this data")
                        ids = [row[0] for row in data]
                        self.database.update_value(table="mmr_data", column_name= "uploaded_eye3", ids=ids)
                    else:     
                        if self.url:
                            # 2a. Send data to EYE3 HTTP
                            ids_success = self.upload_http(data, self.url)
                            # 2b. Update column uploaded_mining with value 1
                            if ids_success: self.database.update_value(table="mmr_data", column_name= "uploaded_eye3", ids=ids_success)

                else:
                    print("No data to upload to EYE3")
            else:
                print("Envio de data a EYE3 esta desactivado")
        
        except:
            self.traceback()


    def upload_mqtt(self, data, topic):
        print(f"Try to upload data to {topic}")
        ids = []
        client = mqtt.connect()
        try:
            if client.is_connected():
                sub_topic = f"acks/{topic}" 
                client.subscribe(sub_topic)        
                for row in data:
                    packet = []
                    for microdata in eval(row[1]):
                        dict_microdata = {
                            "measurement": {
                                "aspersores": " ".join(str(x) for x in microdata["valve_state"]),
                                "f_l": microdata["f_l"],
                                "sq_l": microdata["sq_l"],
                                "v_l": microdata.get("v_l", -1),
                                "f_r": microdata["f_r"],
                                "sq_r": microdata["sq_r"],
                                "v_r": microdata.get("v_r", -1),
                                "latitude": microdata["latitude"],
                                "longitude": microdata["longitude"],
                                "speed_kmh": microdata["speed"],
                                "geofence_id": microdata["zone_id"],
                            },
                            "time" : microdata["timestamp"]
                        }
                        packet.append(dict_microdata)

                    dict_data = {
                        "packet": packet,
                        "id": row[0],
                        "name": self.machine_name,
                        "machineid": int(self.machine_id)
                    }
                    print(f"Data to upload: {dict_data}")

                    result = mqtt.publish(client= client, topic=topic, dict_data=dict_data)
                    if result:
                        print(f"Data uploaded Success to {topic}")
                        res = mqtt.get_response(wait=2)
                        if res:
                            if row[0] == res.get("id", -1) and res.get("status", "error") == "success": ids.append(row[0])
                            else: print("Error al escribir en la base de datos del server")
                        else: print("No hay respuesta del server")
                        
                client.loop_stop()
                client.disconnect()
                return ids
            else:
                print("No se logró conectar a MQTT")
        except:
            self.traceback()
        return ids



    def upload_http(self, data, url):
        print(f"Try to upload data to {url}")
        ids = []
        for row in data:
            puntos, current_irrigation_rate = self.process_data_eye3(row[1])
            # Only if there are points to upload
            if puntos:
                jsonitem = {
                        'idpoligono': 0,  # Al ver las tablas y los scripts siempre es 0
                        'tasaideal': current_irrigation_rate,  # Este valor si cambia, puede ser 0,1,2
                        'horaingreso': puntos[0]["date"],
                        'horasalida': puntos[-1]["date"],
                        'pm10': None, # al ver las tablas siempre es None
                        'pm25': None, # al ver las tablas siempre es None
                        'velocidadpromedio': row[4], # Este valor si cambia
                        'vehiculo': self.machine_id,
                        'puntos': puntos
                        }

                try:
                    print(f"Data to upload: {jsonitem}")
                    r = requests.post(url= url,  data = {'json': json.dumps(jsonitem)}, timeout=2)
                    content = r.content.decode("utf-8")
                    print(content)
                    if '''{"ok":true}''' in content:
                        ids.append(row[0])
                        print("row number uploaded successfully to %s", url)
                    else:
                        print("Upload FAILED. Will retry in  seconds")
                        sleep(1)

                except requests.exceptions.ConnectionError:
                    print("Failed to upload to %s [ConnectionError]" % (url))

                except Exception as err:
                    print("Failed to upload to %s [unknown error]" % (url))
                    print(repr(err))
                sleep(0.5)
            

            else:
                # si no hay puntos significa que solo tenia datos de flujometro, pero eye3 no necesita esos datos, asi que ignoramos esa fila para eye3
                ids.append(row[0])
        return ids

    def process_data_eye3(self, row):
        WET_SEGMENT_SCALE_FACTOR = 1.4
        puntos = []
        current_irrigation_rate = 0
        irrigation_disabled = False
        METROS_REGADOS_MANUAL = 0
        for microdata in eval(row):
            # {"dist": 5.023454, "rtc": 0, "infraction": 0, "forbidden": 0, "lat": -22.315512, "rtr": 0, 
            # "date": "2025-01-22 20:58:02", "speed": 18.04774, "ltr": 0, "lon": -68.902291, "ltc": 0, 
            # "truck": 888, "mode": "MANUAL"}}
            valve_state = microdata["valve_state"]
            if valve_state != [-1, -1, -1, -1] and microdata["latitude"] !=-1 and microdata["longitude"] !=-1 : # Si no es un dato erroneo se envia a eye3, si es erroneo agregar id para dejarlo como subido
                if microdata['irrigation_type'] == "GEOCERCO":
                    irrigation_disabled= False

                elif microdata['irrigation_type'] == "NO_REGAR":
                    irrigation_disabled= True

                elif microdata['irrigation_type'] == "RIEGO_SEGMENTADO":
                    irrigation_disabled= False

                elif microdata['irrigation_type'] == "RIEGO_CONTINUO":
                    irrigation_disabled= False
                
                if microdata['road_type'] == "RAMPA" and irrigation_disabled == False:
                    current_irrigation_rate = 2

                distance_from_last_point = microdata["distance"] if microdata["distance"]<100000 else 0
                segmentos = (0,0)
                if "-" in microdata["pattern"]:
                    segmentos = microdata["pattern"].split("-")
                m_reg = segmentos[0]
                json_infraccion_no_regar = 0
                json_infraccion_velocidad_min = 0
                json_infraccion_velocidad_max = 0
                valve_state = microdata["valve_state"]
                ltr, ltc, rtc, rtr = valve_state
                json_data ={
                    "truck": str(self.machine_id),
                    "lat": round(microdata["latitude"],6),
                    "lon": round(microdata["longitude"],6),
                    "speed": microdata["speed"],
                    "dist": microdata["distance"],
                    "infraction": 0, # Que es esto? puede ser [0,1,2,3,4,5]
                    "forbidden": 0,    # Es 0 segun script __main__ riego_aspersor
                    "ltr": valve_state[0],
                    "ltc": valve_state[1],
                    "rtc": valve_state[2],
                    "rtr": valve_state[3],
                    "mode": "MANUAL",
                    "date": datetime.fromtimestamp(int(microdata["timestamp"]/1000)).strftime('%Y-%m-%d %H:%M:%S'),
                }
                rtr_should_be_on = 1
                rtc_should_be_on = 1
                ltc_should_be_on = 1
                ltr_should_be_on = 1
                if microdata['irrigation_type'] == "NO_REGAR":
                    rtr_should_be_on = 0
                    rtc_should_be_on = 0
                    ltc_should_be_on = 0
                    ltr_should_be_on = 0
                else:
                    rtr_should_be_on = microdata['rtr_should_be'] if microdata['rtr_should_be'] else 1 # set en 1 if not geofence
                    rtc_should_be_on = microdata['rtc_should_be'] if microdata['rtc_should_be'] else 1 # set en 1 if not geofence
                    ltc_should_be_on = microdata['ltc_should_be'] if microdata['ltc_should_be'] else 1 # set en 1 if not geofence
                    ltr_should_be_on = microdata['ltr_should_be'] if microdata['ltr_should_be'] else 1 # set en 1 if not geofence

                if ltr or ltc or rtc or rtr:
                    # Chequeo de riego en zona prohibida.
                    # De paso, en zonas de tipo NO_REGAR, prohibimos forzosamente el uso de aspersores
                    if microdata["irrigation_type"]== "NO_REGAR":
                        json_infraccion_no_regar = 1

                    # Chequeos de velocidad se realizan unicamente en geocercos conocidos
                    if microdata['zone_id'] != -1:
                        # Chequeo de riego a exceso de velocidad
                        if microdata["speed"] > microdata["min_speed"]:
                            json_infraccion_velocidad_max = 1
                        else:
                            json_infraccion_velocidad_max = 0

                        # Chequeo de riego bajo el limite minimo de velocidad
                        if microdata["speed"] < microdata["max_speed"]:
                            json_infraccion_velocidad_min = 1
                        else:
                            json_infraccion_velocidad_min = 0

                if (rtc + ltc) > (rtc_should_be_on + ltc_should_be_on):
                    json_infraccion_aspersores_centrales = True
                else:
                    json_infraccion_aspersores_centrales = False

                if (rtr + ltr) > (rtr_should_be_on + ltr_should_be_on):
                    json_infraccion_aspersores_laterales = True
                else:
                    json_infraccion_aspersores_laterales = False
                

                if microdata['zone_id'] != -1:
                    if rtr== 1 or rtc == 1 or ltc == 1 or ltr == 1:
                        METROS_REGADOS_MANUAL = METROS_REGADOS_MANUAL + distance_from_last_point
                        print("METROS_REGADOS_MANUAL = [%s]" % METROS_REGADOS_MANUAL)
                        if float(METROS_REGADOS_MANUAL) > (float(m_reg) * WET_SEGMENT_SCALE_FACTOR):
                            if microdata['irrigation_type'] == "RIEGO_SEGMENTADO":
                                json_infraccion_segmentado = 1
                            else:
                                json_infraccion_segmentado = 0
                        else:
                            json_infraccion_segmentado = 0
                    else:
                        json_infraccion_segmentado = 0
                        METROS_REGADOS_MANUAL = 0
                else:
                    json_infraccion_segmentado = 0

                
                        
                string_code = ""
                if json_infraccion_no_regar:
                    string_code = string_code + "1"


                if json_infraccion_velocidad_min:
                    string_code = string_code + "2"

                if json_infraccion_aspersores_centrales or json_infraccion_aspersores_laterales:
                    string_code = string_code + "3"

                if json_infraccion_velocidad_max:
                    string_code = string_code + "4"

                if json_infraccion_segmentado:
                    string_code = string_code + "5"

                if string_code == "":
                    infraction_code = 0
                else:
                    infraction_code = int(string_code)
                
                json_data["infraction"] = infraction_code

                puntos.append(json_data)
        return puntos, current_irrigation_rate

if __name__ == "__main__":
    print("---")
    print("INIT AUTOUPLOAD")
    upload = Upload(CONFIG_SOURCE) 
    while True:
        try:
            upload.check_data()
        except:
            print("error")
            e = sys.exc_info()
            print("dumping traceback for [%s: %s]" % (str(e[0].__name__), str(e[1])))
            traceback.print_tb(e[2])
            foo = "bar" # do nothing
        sleep(10)
        
