import threading
from time import time, sleep
from lib.utils import Utils
from lib.minimalmodbus import Instrument, MODE_RTU
import serial
from struct import pack, unpack

class FlowLib(Utils):
    def __init__(self, usbdevnode, modbus_id, baudrate: int = 9600, log_id = "FLOW") -> None:
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.port = usbdevnode.get_devnode()
        self.modbus_id = modbus_id
        self.last_flow_timestamp = time()
        self.instrument = []
        self.connect()
        threading.Thread(target = self.read).start()

    def connect(self) -> None:
        try:
            for i in range(2):
                client = Instrument(self.port, self.modbus_id[i], close_port_after_each_call=True, debug=False)
                client.serial.baudrate = 9600
                self.instrument.append(client)
        except:
            self.traceback()
            sleep(1)

    def read_float_reg(self,regOne, regTwo, instrument):
        data = (instrument.read_register(
            regOne), instrument.read_register(regTwo))
        packed_string = pack("HH", *data)
        unpacked_string = unpack("f", packed_string)[0]
        return float("{:.2f}".format(unpacked_string))

    def read_flow(self,instrument):
        flow = -1
        try:
            # Convert m3/h to l/s
            flow = self.read_float_reg(0, 1, instrument) * 1000 / 3600
            print(f'Flow rate: {flow} l/s')
        except:
            print(f"Error reading flow")
        return flow

    def read_signal_quality(self, instrument):
        quality = -1
        try:
            return(instrument.read_register(91)& 0x00FF)
        except:
            print(f"Error reading quality")
        return quality


    def read_volume(self,instrument):
        volume = -1
        try:
            volume = self.read_float_reg(112, 113, instrument)
            print(f'Volume: {volume} m3')
        except:
            print(f"Error reading volume")
        return volume



    def read_flowmeter(self, instrument):
        flow, quality, volume = -1, -1, -1
        try:
            # 1. Read flow
            flow = self.read_flow(instrument)
            sleep(0.3)

            # 2. Read Quality
            quality = self.read_signal_quality(instrument)
            sleep(0.3)

            # 3. Read Volume
            volume = self.read_volume(instrument)
            sleep(0.3)


            self.log(f"Flow l/s = {flow} - Flow m3/h = {flow*3.6} - Quality={quality} - Volume m3={volume} - flowmeter modbus id: {instrument.address}")
            self.last_flow_timestamp = time()

            return flow, quality, volume
        except:
            self.log(f"Error reading flowmeter modbus id: {instrument.address}")
        
        return flow, quality, volume
    
    def read(self) -> None:
        while True:
            try:
                # Read Left Flowmeter
                flow_left, quality_left, volume_left = self.read_flowmeter(self.instrument[0])
                sleep(0.4)

                # Read Right Flowmeter
                flow_right, quality_right, volume_right = self.read_flowmeter(self.instrument[1])
                sleep(0.4)

                # Proccess
                self.proccess_line(flow_left, flow_right, quality_left, quality_right, volume_left, volume_right)
            except:
                self.traceback()
                sleep(1)
            

    def proccess_line(self, flow_left, flow_right, sq_left, sq_right, volume_left, volume_right) -> None:
        try:
            data_dict = {
                "f_l": flow_left,
                "sq_l": sq_left,
                "v_l": volume_left,
                "f_r": flow_right,
                "sq_r": sq_right,
                "v_r": volume_right,
                "timestamp": int(time()*1000)
            }
            self.log(f"Flow Data: {data_dict}")
            self.emit(data_type=self.log_id, data=data_dict)

        except:
            self.traceback() 
               


    

