import json
import socket
import sqlite3
import threading
from time import sleep
from datetime import datetime, timedelta
import sys

sys.path.append('/srv/datalogger_gimp/')
from lib.utils import Utils
from database.models import Database

class Server(Utils):
    def __init__(self, ip, port, mode, log_id="SERVER"):
        super().__init__(log_id)
        self.mode = mode
        self.acum_pm100, self.acum_pm10 = [], []
        self.last_latitude = 0
        self.last_longitude = 0
        self.localIP, self.localPort, self.bufferSize = ip, port, 1024
        self.UDPServerSocket = self.initialize_socket()
        self.next_save_time = self.calculate_next_save_time()
        self.database = Database()

        threading.Thread(target=self.read_client_data).start()

    def initialize_socket(self):
        """Inicializa el socket UDP."""
        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        sock.bind((self.localIP, self.localPort))
        self.log(f"Server listening on {self.localIP}:{self.localPort}")
        return sock

    def calculate_next_save_time(self):
        """Calcula el próximo intervalo de guardado en los minutos 0, 15, 30 o 45."""
        dt = datetime.now()
        return (dt + timedelta(minutes=(15 - dt.minute % 15))).replace(second=0, microsecond=0)
  
    def read_client_data(self):
        """Lee datos de los clientes, procesa los mensajes y guarda promedios en intervalos."""
        self.log("Server is running and waiting for data...")
        
        while True:
            try:
                # Recibir datos del cliente
                bytesAddressPair = self.UDPServerSocket.recvfrom(self.bufferSize)
                message = bytesAddressPair[0].decode("utf-8")
                self.log(f"Received: {message}")
                msg = json.loads(message)
                self.process_data(msg.get('name_id', 'UNKNOWN'), msg)

                # Procesar guardado de promedios al alcanzar el siguiente intervalo
                if datetime.now() >= self.next_save_time:
                    self.save_averages()
                    self.reset_accumulated_data()
                    self.next_save_time = self.calculate_next_save_time()

                sleep(0.01)
            except Exception as ex:
                self.log(f"Error reading data: {ex}")
                self.traceback()

    def save_averages(self):
        """Calcula e inserta los promedios en la base de datos."""
        pm100_avg = self.calculate_average(self.acum_pm100)
        pm10_avg = self.calculate_average(self.acum_pm10)
        
        self.log(f"Saving averages: PM100: {pm100_avg}, PM10: {pm10_avg}, Latitud: {self.last_latitude}, Longitud: {self.last_longitude}")
        if not self.mode: #No esta en modo test, por lo que debe guardar datos
            self.database.insert_gimp_data(pm100_avg, pm10_avg, self.last_latitude, self.last_longitude)
        else:
            self.log("Cuidado equipo en modo test, recuerde pasar a modo normal despues de pruebas")

    def reset_accumulated_data(self):
        """Resetea las variables de acumulación."""
        self.acum_pm100.clear()
        self.acum_pm10.clear()
        self.last_latitude = 0 
        self.last_longitude = 0
    

    def calculate_average(self, data):
        """Calcula el promedio de una lista de datos."""
        return round(sum(data) / len(data), 2) if data else None

    def calculate_averages(self, data_store):
        """Calcula los promedios de cada clave en el diccionario data_store."""
        return {key: self.calculate_average(values) for key, values in data_store.items()}

    def process_data(self, name_id, payload):
        """Procesa los datos recibidos dependiendo del tipo."""
        if name_id == "SERIAL":
            self.acum_pm100.append(payload["data"].get("pm100", 0))
            self.acum_pm10.append(payload["data"].get("pm10", 0))
        
        elif name_id == "GPS_USB":
            self.last_latitude = payload["data"].get("latitude", 0)
            self.last_longitude = payload["data"].get("longitude", 0)
            self.log(f"Processed GPS data: {self.last_latitude}, {self.last_longitude}")

        else:
            self.log(f"Unknown message type: {name_id}")


# Configuración
with open('/srv/datalogger_gimp/config_gimp.json') as f:
    config = json.load(f)

if __name__ == "__main__":
    Server(
        ip=config["SERVER"]["IP"],
        port=config["SERVER"]["PORT"],
        mode = config["TEST_MODE"]
    )
