import json
import serial
import socket
import threading
import json
from time import sleep, time
from datetime import datetime
import sys

sys.path.append('/srv/datalogger_mmr/')
from lib.utils import Utils
from database.models import Database


class Server(Utils):
    def __init__(self, max_len_packet_data, seconds_microdata, ip, port, mode, log_id = "SERVER"):
        self.log_id = log_id
        self.mode = mode
        self.packet_data = []
        self.max_len_packet_data = max_len_packet_data
        self.seconds_microdata = seconds_microdata
        self.timer_microdata = serial.serialutil.Timeout(self.seconds_microdata)
        self.flag_flow = False
        self.count = 0
        self.acum_speed = []
        self.acum_valve_state = [0, 0, 0, 0]
        self.local_ip = ip
        self.local_port = port
        self.buffer_size = 1024
        self.database = Database()
        self.UDPServerSocket = socket.socket(family=socket.AF_INET, type=socket.SOCK_DGRAM)
        self.UDPServerSocket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.UDPServerSocket.bind((self.local_ip, self.local_port))
        self.reset_microdata()
        threading.Thread(target=self.read_client_data).start()

    def read_client_data(self)-> None:
        """_summary_
        """
        self.log(f"Server init listening [%s port %s]" % (self.local_ip, self.local_port))
        while True:
            try:
                # Wait from msg clients
                bytesAddressPair = self.UDPServerSocket.recvfrom(self.buffer_size)
                message = bytesAddressPair[0].decode('utf-8')
                msg =  json.loads(message)
                #print(msg)
                name_id = msg['name_id']
                self.process_data(name_id, msg)

                # Wait for flow or valve states
                if self.timer_microdata.expired():
                    microdata = self.construct_microdata()
                    if self.flag_flow:
                        self.packet_data.append(microdata)
                        self.log(f"dato numero {self.count} : {microdata}")
                        self.count = self.count + 1
                        for i in range(len(microdata["valve_state"])):
                            if microdata["valve_state"][i]!= -1: self.acum_valve_state[i] += microdata["valve_state"][i]
                                

                    dict_data = {
                        "latitude": microdata["latitude"],
                        "longitude": microdata["longitude"],
                        "speed": microdata["speed"],
                        "zone_id": microdata["zone_id"],
                        "timestamp": microdata["timestamp"],
                        "valve_state": 1 if 1 in microdata["valve_state"] else 0, # TODO: aca considerar si el flujometro esta marcando dato
                    }
                    
                    with open("/data_live.json", "w") as f:
                        f.write(json.dumps(dict_data))

                    # Restart variables and timer
                    self.reset_microdata()
                    self.timer_microdata.restart(self.seconds_microdata)

                # Wait until complete n packet data
                if self.count == self.max_len_packet_data:
                    if self.packet_data:
                        self.log("Packet data")
                        avg_speed = self.calculate_average(self.acum_speed)
                        if not self.mode: #No esta en modo test, por lo que debe guardar datos
                            self.database.insert_mmr_data(self.packet_data, avg_speed, self.acum_valve_state)
                        else:
                            self.log("Cuidado equipo en modo test, recuerde pasar a modo normal despues de pruebas")

                    # Restart Variables
                    self.packet_data = []
                    self.count = 0
                    self.acum_speed = []
                    self.acum_valve_state = [0, 0, 0, 0]
                sleep(0.01)

            except Exception as ex:
                self.traceback()

    def reset_microdata(self)-> None:
        # Reset all values to -1
        self.last_flow_data = {"f_l": -1, "sq_l": -1, "v_l": -1, "f_r": -1, "sq_r": -1, "v_r": -1}
        self.last_valve_state = {"valve_state":[-1, -1, -1, -1]}
        self.last_gps_data = {  "latitude": -1, 
                                "longitude": -1, 
                                "speed": -1,
                                'distance': -1,
						        'angle': -1,
                                "zone_id": -1,
                                "road_type": "UNKNOWN",
                                "location": "UNKNOWN"}
        self.flag_flow = False
    
    def construct_microdata(self)-> dict:
        microdata = {}
        microdata.update(self.last_flow_data)
        microdata.update(self.last_valve_state)
        microdata.update(self.last_gps_data)
        microdata["timestamp"] = int(time()*1000)

        return microdata

    def process_data(self, name_id: str, payload: dict)-> None:
        """_summary_

        Args:
            name_id (str): _description_
            payload (dict): _description_
        """
        if name_id == "SERIAL":
            self.flag_flow = True
            self.last_valve_state.update(payload["data"])

        elif name_id== "GPS_USB":
            self.acum_speed.append(payload["data"]["speed"])
            self.last_gps_data.update(payload["data"])

        elif name_id == "FLOW":
            self.flag_flow = True
            self.last_flow_data.update(payload["data"])

        else:
            print(f"That's not a valid type: {name_id}")

    def calculate_average(self, data):
        """Calcula el promedio de una lista de datos."""
        return round(sum(data) / len(data), 2) if data else None

    


# Config
f = open('/srv/datalogger_mmr/config_mmr.json')
config:dict = json.load(f)

if __name__ == "__main__":
	Server( max_len_packet_data= config["SERVER"]["MAX_LEN_PACKET_DATA"], 
            seconds_microdata= config["SERVER"]["SECONDS_MICRODATA"],
            ip=config["SERVER"]["IP"], 
            port=config["SERVER"]["PORT"],
            mode = config["TEST_MODE"]
            )
