import threading
from serial import Serial
from time import time, sleep
from lib.utils import Utils

class SerialLib(Utils):
    def __init__(self, usbdevnode, level_curve, config, baudrate: int = 38400, log_id = "SERIAL") -> None:
        self.config = config
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.usbdevnode = usbdevnode
        self.m, self.b = level_curve # lista con coeficientes de la curva mx+b --> [m, b]
        self.last_valve_timestamp = time()
        
        threading.Thread(target = self.connect).start()

    def connect(self) -> None:
        """
        This function attempts to establish a serial connection with the specified USB device node.

        """

        try:
            self.log(f"Try to connect serial port: {self.usbdevnode.get_devnode()}")
            self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=self.timeout)
            self.read()

        except Exception as Ex:
            self.log(Ex)

    def read(self) -> None:
        """
        This function continuously reads lines from the serial module and processes them.
        If the line is empty, the function skips it.

        """
        # modo:0;extra:0;ltr:0;ltc:0;rtr:0;rtc:0;m_extra:1;m_ltr:0;m_ltc:1;m_rtr:1;m_rtc:1;a_extra:1;a_ltr:0;a_ltc:1;a_rtr:1;a_rtc:1;templm:0.0
        self.log("Reading line from serial")
        while True:
            try:
                raw_line = self.serial_module.readline().decode("utf-8")
                line = raw_line.strip()
                if line =="":
                    pass
                else:
                    self.log(f"GOT: {line}")
                    if "modo" in line:self.process_line(line)
                sleep(0.01)
            except:
                self.log("Error decoding line")

    def process_line(self, line: str) -> None:
        try:
            data_dict = {}
            data_split = line.split(';')
            for data in data_split:
                if not data: continue
                data = data.upper()
                if "M_" in data and not "M_EXTRA" in data: 
                    self.log(data)
                    serial_var, serial_value = data.split(':')
                    data_dict[serial_var] = int(float(serial_value))

                if "EXTRA_0" in data or "EXTRA_1" in data or "EXTRA_2" in data: 
                    self.log(data)
                    serial_var, serial_value = data.split(':')
                    data_dict[serial_var] = float(serial_value)
                

            self.log(data_dict)
            valve_state = {
                "aspersores": [data_dict["M_LTR"], data_dict["M_LTC"], data_dict["M_RTC"], data_dict["M_RTR"]],
                "m_ltr": data_dict["M_LTR"],
                "m_ltc": data_dict["M_LTC"],
                "m_rtc": data_dict["M_RTC"],
                "m_rtr": data_dict["M_RTR"],
                "timestamp": time()
            }
            if "EXTRA_0" in data_dict and "EXTRA_1" in data_dict and "EXTRA_2" in data_dict:
                valve_state["extra_0"] = data_dict["EXTRA_0"]
                valve_state["extra_1"] = data_dict["EXTRA_1"]
                valve_state["extra_2"] = data_dict["EXTRA_2"]
                
            if  self.m != 0 or self.b != 0:
                valve_state["level"] = (self.m*data_dict["EXTRA_1"]) + self.b
            self.emit(data_type=self.log_id, data=valve_state)
            self.last_valve_timestamp = time()
            

        except:
            self.traceback() 
               

