let SHOW_LEVEL = false;
let SHOW_VOLUME = false;

async function cargarConfigHUD() {
  try {
    const res = await fetch("/config");
    if (!res.ok) throw new Error("No se pudo cargar /config");

    const cfg = await res.json();

    SHOW_LEVEL = cfg.SHOW_LEVEL;
    SHOW_VOLUME = cfg.SHOW_VOLUME;

    // Controlar visibilidad inmediata según flags
    document.getElementById("hud-nivel-card").style.display = SHOW_LEVEL
      ? "block"
      : "none";

    document.getElementById("hud-volumen-card").style.display = SHOW_VOLUME
      ? "block"
      : "none";

    console.log(
      `HUD CONFIG → SHOW_LEVEL=${SHOW_LEVEL} | SHOW_VOLUME=${SHOW_VOLUME}`
    );
  } catch (err) {
    console.error("Error cargando configuración del HUD:", err);
    // Por seguridad, ocultar ambos HUD si falla la config
    document.getElementById("hud-nivel-card").style.display = "none";
    document.getElementById("hud-volumen-card").style.display = "none";
  }
}

// =========================
// Actualización del HUD
// =========================
// Función para actualizar el nivel del estanque
async function actualizarNivel() {
  //Si está deshabilitado desde /config → ocultar y salir
  if (!SHOW_LEVEL) {
    document.getElementById("hud-nivel-card").style.display = "none";
    return;
  }

  try {
    const nivelRes = await fetch("/nivel");
    if (!nivelRes.ok) throw new Error("No se pudo consultar /nivel");
    const nivelData = await nivelRes.json();

    const hudNivelCard = document.getElementById("hud-nivel-card");
    const nivelBar = document.getElementById("hud-nivel-bar");
    const nivelText = document.getElementById("hud-nivel-text");
    const volumenSpan = document.getElementById("hud-volumen-estanque");

    // Verificar si el timestamp Unix es mayor a 2 días
    timestampUnix = nivelData.timestamp; // Asumiendo que "timestamp" es en formato Unix
    if (timestampUnix) {
      const timestamp = new Date(timestampUnix * 1000); // Convertir Unix a milisegundos
      const now = new Date();
      const diffInDays = (now - timestamp) / (1000 * 3600 * 24); // Diferencia en días

      if (diffInDays > 2) {
        // Si han pasado más de 2 días, ocultamos las viñetas y mostramos el mensaje en consola
        hudNivelCard.style.display = "none";
        console.log("NIVEL DESACTUALIZADO, TIMESTAMP MAYOR A 2 DIAS");
        return; // Detener la ejecución si el timestamp es mayor a 2 días
      }
    }

    // Si el timestamp es válido y no han pasado más de 2 días

    // Verificar si los datos del nivel son válidos
    if (nivelData.porcentaje == null) {
      hudNivelCard.style.display = "none";
    } else {
      hudNivelCard.style.display = "block";
      const porcentaje = nivelData.porcentaje ?? 0;
      const volumenEstanque = nivelData.volumen_estanque ?? 0;

      // 🎨 Color dinámico: rojo <50%, azul >=50%
      let color;
      if (porcentaje < 50) {
        color = "#e74c3c"; // rojo alerta
      } else {
        color = "#3498db"; // azul mining
      }

      // Actualizar la barra y texto
      nivelBar.style.width = `${porcentaje}%`;
      nivelBar.style.background = `linear-gradient(90deg, ${color}, ${color})`;
      nivelText.textContent = `${porcentaje}%`;
      volumenSpan.textContent = `${volumenEstanque} m³`;
    }
  } catch (err) {
    console.error("Error al actualizar nivel:", err);
    document.getElementById("hud-nivel-card").style.display = "none";
  }
}

// Función para actualizar el volumen acumulado
async function actualizarVolumen() {
  // Si está deshabilitado desde /config
  if (!SHOW_VOLUME) {
    document.getElementById("hud-volumen-card").style.display = "none";
    return;
  }
  try {
    const volRes = await fetch("/volumen");
    if (!volRes.ok) {
      document.getElementById("hud-volumen-card").style.display = "none";
      throw new Error("No se pudo consultar /volumen");
    }
    const volData = await volRes.json();
    const hudVolumenCard = document.getElementById("hud-volumen-card");

    // Verificar si el timestamp Unix es mayor a 2 días
    timestampUnix = volData.timestamp; // Asumiendo que "timestamp" es en formato Unix
    if (timestampUnix) {
      const timestamp = new Date(timestampUnix * 1000); // Convertir Unix a milisegundos
      const now = new Date();
      const diffInDays = (now - timestamp) / (1000 * 3600 * 24); // Diferencia en días

      if (diffInDays > 2) {
        // Si han pasado más de 2 días, ocultamos las viñetas y mostramos el mensaje en consola
        hudVolumenCard.style.display = "none";
        return; // Detener la ejecución si el timestamp es mayor a 2 días
      }
    }

    if (volData == null) {
      hudVolumenCard.style.display = "none";
    } else {
      hudVolumenCard.style.display = "block";
    }

    const volumenAcum = volData.volumen ?? 0;
    const volumenTime = volData.timestamp
      ? new Date(volData.timestamp * 1000).toLocaleTimeString([], {
          hour: "2-digit",
          hour12: false,
        }) + " h"
      : "--";

    document.getElementById("hud-volumen-value").textContent =
      volumenAcum.toLocaleString();
    document.getElementById("hud-volumen-time").textContent = volumenTime;
  } catch (err) {
    console.error("Error al actualizar volumen:", err);
    document.getElementById("hud-volumen-card").style.display = "none";
  }
}

// =========================
// Velocidad del Algibe
// =========================
async function getVelocidad() {
  console.log("Obteniendo velocidad del algibe...");
  try {
    // Obtiene el ID localmente
    const resConfig = await fetch("/data");
    if (!resConfig.ok) throw new Error("No se pudo leer /data");
    const config = await resConfig.json();

    const configuracion = await fetch("/config");
    const data = await configuracion.json();

    API_URL_REG = data.API_URL_REG;
    const ALGIBE_ID = "REG-" + config.name;

    console.log("Configuración cargada:");
    console.log("API_URL_REG:", API_URL_REG);
    console.log("ALGIBE_ID:", ALGIBE_ID);

    const titulo = document.querySelector(".hud-title");
    if (titulo) {
      titulo.textContent = ALGIBE_ID;
    }

    if (!API_URL_REG || !ALGIBE_ID) {
      console.error("Falta configuración en .json");
      return null;
    }

    //Obtiene la lista de vehículos
    const res = await fetch(API_URL_REG);
    if (!res.ok) throw new Error("No se pudo obtener lista de vehículos");
    const lista = await res.json();
    // Busca el vehículo configurado
    const vehiculo = lista.find((v) => v.vehiculo === ALGIBE_ID);
    if (!vehiculo) {
      console.warn(`No se encontró el vehículo ${ALGIBE_ID}`);
      return null;
    }

    // Retorna solo la velocidad
    return vehiculo.speed_kmh ?? 0;
  } catch (error) {
    console.error("Error al obtener velocidad:", error);
    return null;
  }
}

async function actualizarVelocidad() {
  const velocidad = await getVelocidad();

  const hudSpeed = document.getElementById("hud-speed");
  const hudStamp = document.getElementById("hud-stamp");

  if (velocidad == null) {
    hudSpeed.textContent = "--";
    hudStamp.textContent = "Sin datos";
    return;
  }

  hudSpeed.textContent = velocidad.toFixed(1);
  hudStamp.textContent = new Date().toLocaleTimeString([], {
    hour: "2-digit",
    minute: "2-digit",
  });
}

// Función para actualizar el título del camión
async function actualizarTituloCamion() {
  try {
    const resConfig = await fetch("/data");
    if (!resConfig.ok) throw new Error("No se pudo leer /data");
    const config = await resConfig.json();

    const ALGIBE_ID = "REG-" + config.name;
    const titulo = document.querySelector(".hud-title");

    if (titulo) {
      titulo.textContent = ALGIBE_ID;
    }
  } catch (err) {
    console.error("Error al actualizar título del camión:", err);
  }
}

async function comprobarBoton() {
  try {
    // Leer el archivo nivel.json
    const res = await fetch("/nivel");

    if (!res.ok) {
      throw new Error("No se pudo leer el archivo /nivel");
    }

    // Convertir el contenido del archivo JSON a un objeto
    const data = await res.json();

    // Verificar si la clave 'button' existe y es true
    const botonVisible = data.button === true;

    // Mostrar u ocultar el botón basado en el valor de 'button'
    const btnReiniciar = document.getElementById("btn-reiniciar-contador");

    if (botonVisible) {
      // Si 'button' es true, mostrar el botón
      btnReiniciar.style.display = "block";
    } else {
      // Si 'button' es false o no existe, ocultar el botón
      btnReiniciar.style.display = "none";
    }
  } catch (err) {
    console.error("Error al comprobar el botón:", err);
  }
}
// Función principal para actualizar todo el HUD
async function actualizarHUD() {
  const title = document.getElementById("title");

  const res = await fetch("/config");
  if (!res.ok) throw new Error("No se pudo cargar /config");

  const cfg = await res.json();

  title.textContent = cfg.TITLE;

  await cargarConfigHUD();
  await actualizarNivel();
  await actualizarVolumen();
  await actualizarTituloCamion();
  await actualizarVelocidad();
  // Llamar a la función mostrarBoton
  comprobarBoton();
}

// Actualiza cada 3 segundos
setInterval(actualizarHUD, 3000);
actualizarHUD();
