import os
import json
import threading
import zoneinfo
from serial import Serial, serialutil
from datetime import datetime
from time import time, sleep
from lib.utils import Utils
from lib.minimalmodbus import Instrument, MODE_RTU
from struct import pack, unpack

class FlowLib(Utils):
    def __init__(self, usbdevnode, model_sensor_level, show_level, show_volume, max_cistern_level, modbus_id, baudrate: int = 9600, log_id = "FLOW") -> None:
        self.baudrate = baudrate
        self.timeout = 0.5
        self.log_id = log_id
        self.port = usbdevnode.get_devnode()
        self.model_sensor_level = model_sensor_level
        self.show_level = show_level
        self.show_volume = show_volume
        self.max_cistern_level = max_cistern_level
        self.modbus_id = modbus_id
        self.last_flow_timestamp = time()
        self.seconds_save_data = 10
        self.timer= serialutil.Timeout(self.seconds_save_data)
        self.last_reference_volume = 0
        self.last_total_volume = 0
        self.instrument = []
        self.connect()
        threading.Thread(target = self.read).start()
        if self.show_level:
            threading.Thread(target = self.read_button).start()

    def connect(self) -> None:
        try:
            for i in range(2):
                client = Instrument(self.port, self.modbus_id[i], close_port_after_each_call=True, debug=False)
                client.serial.baudrate = 9600
                self.instrument.append(client)
        except:
            self.traceback()
            sleep(1)

    def read_float_reg(self,regOne, regTwo, instrument):
        data = (instrument.read_register(
            regOne), instrument.read_register(regTwo))
        packed_string = pack("HH", *data)
        unpacked_string = unpack("f", packed_string)[0]
        return float("{:.2f}".format(unpacked_string))

    def read_flow(self,instrument):
        flow = None
        try:
            # Convert m3/h to l/s
            flow = self.read_float_reg(0, 1, instrument) * 1000 / 3600
            print(f'Flow rate: {flow} l/s')
        except:
            print(f"Error reading flow")
        return flow

    def read_signal_quality(self, instrument):
        quality = None
        try:
            return(instrument.read_register(91)& 0x00FF)
        except:
            print(f"Error reading quality")
        return quality


    def read_volume(self,instrument):
        volume = None
        try:
            volume = self.read_float_reg(112, 113, instrument)
            print(f'Volume: {volume} m3')
        except:
            print(f"Error reading volume")
        return volume



    def read_flowmeter(self, instrument):
        flow, quality, volume = None, None, None
        try:
            # 1. Read flow
            flow = self.read_flow(instrument)
            sleep(0.3)

            # 2. Read Quality
            quality = self.read_signal_quality(instrument)
            sleep(0.3)

            # 3. Read Volume
            volume = self.read_volume(instrument)
            sleep(0.3)


            self.log(f"Flow l/s = {flow} - Flow m3/h = {flow*3.6} - Quality={quality} - Volume m3={volume} - flowmeter modbus id: {instrument.address}")
            self.last_flow_timestamp = time()

            return flow, quality, volume
        except:
            self.log(f"Error reading flowmeter modbus id: {instrument.address}")
        
        return flow, quality, volume
    
    def read(self) -> None:
        while True:
            try:
                # Read Left Flowmeter
                flow_left, quality_left, volume_left = self.read_flowmeter(self.instrument[0])
                sleep(0.4)

                # Read Right Flowmeter
                flow_right, quality_right, volume_right = self.read_flowmeter(self.instrument[1])
                sleep(0.4)

                # Proccess
                self.proccess_line(flow_left, flow_right, quality_left, quality_right, volume_left, volume_right)
            except:
                self.traceback()
                sleep(1)
            

    def proccess_line(self, flow_left, flow_right, sq_left, sq_right, volume_left, volume_right) -> None:
        try:
            # 1. Construct dict data
            data_dict = {
                "f_l": flow_left,
                "sq_l": sq_left,
                "v_l": volume_left,
                "f_r": flow_right,
                "sq_r": sq_right,
                "v_r": volume_right,
                "timestamp": int(time()*1000)
            }

            # 2. Delete none values and send filter data to local server
            filter_dict = {key: value for key, value in data_dict.items() if value is not None}
            self.log(f"Flow Data: {filter_dict}")
            self.emit(data_type=self.log_id, data=filter_dict)

            # 3. Save data to use in map web
            # 3.1 Truncar a 0 si la medición de volumen fallo y aplicarle valor abs
            volume_left = 0 if volume_left==None else volume_left
            volume_right = 0 if volume_right==None else volume_right
            self.last_total_volume = int(volume_left + volume_right)

            # 3.2 Save data every 10 seconds
            if self.timer.expired():
                # Only calculate volume if flag if true
                if self.show_volume:
                    self.save_volume_data(self.last_total_volume)

                self.timer.restart(self.seconds_save_data)

                # Only calculate level if flag if true and sensor config is tuf
                if self.show_level and self.model_sensor_level == "TUF":
                    self.save_calculate_level_data(self.last_total_volume)

        except:
            self.traceback() 

    def save_calculate_level_data(self, total_volume):
        file_path = "/var/www/html/nivel.json"
        try:
            # 1. Calculate current level
            irrigated_volume = int(total_volume - self.last_reference_volume)
            if irrigated_volume < 0 : irrigated_volume = 0
            elif irrigated_volume > 70: irrigated_volume = 70
            level = int((self.max_cistern_level - irrigated_volume)*100/self.max_cistern_level)
            self.log(f"NIVEL_ESTANQUE_ESTIMADO: {level}%")

            # 2. Save data
            level_data = {
                "porcentaje": level,
                "volumen_estanque": int(self.max_cistern_level*level/100), 
                "timestamp": int(time()),
                 "button": True

            }
            self.log(level_data)
            self.write_file("/var/www/html/nivel.json", json.dumps(level_data), "w")
            
        except Exception as ex:
            print(f"Error calculate level data: {ex}")

    def read_button(self):
        file_path = "/var/www/html/estado_boton.json"
        while True:
            try:
                if os.path.isfile(file_path):
                    estado_botton = json.load(open(file_path))
                    last_timestamp_press_button = estado_botton["timestamp"]
                    # Check if existe a difference max 10 seconds
                    if last_timestamp_press_button > time()-10:
                        # El boton fue presionando recientemente
                        self.last_reference_volume = self.last_total_volume
                        self.timer.restart(0.1) # Restart timer para que reinicie de inmediato el dato
                        print("Se presiono el boton")
                        sleep(10)
                    sleep(1)
            except Exception as ex:
                print(f"Error reading button state tablet: {ex}")

    def save_volume_data(self, total_volume):
        file_path = "/var/www/html/volumen_acumulado.json"
        try:
            # 1. Check if path exist, else create 
            if not os.path.isfile(file_path):
                self.create_base_volume_data(file_path)
                
            # 2. Open current file to check datetime
            try:
                keys_to_check = ["volumen", "last_accumulator_volume", "timestamp", "datetime_cl"]
                volume_data = json.load(open(file_path))

                # Si no estan todas las keys presentes, crear el archivo nuevamente
                if not set(keys_to_check).issubset(volume_data.keys()): self.create_base_volume_data(file_path)
            
            except Exception as ex:
                print(f"Error, archivo probablemente corrupto, creandolo nuevamente {ex}")
                self.create_base_volume_data(file_path)
                
            # 3. Transformar file datetime to object
            santiago_tz = zoneinfo.ZoneInfo("America/Santiago")
            datetime_str_cl = volume_data["last_accumulator_volume"][2]
            datetime_file_cl = datetime.strptime(datetime_str_cl, '%Y-%m-%d %H:%M:%S')

            # 4. Get current datetime
            datetime_now_utc = datetime.now()
            timestamp = int(datetime_now_utc.timestamp())
            datetime_now_cl = datetime_now_utc.astimezone(santiago_tz)
            print(f'Volume file datetime: {datetime_file_cl} - Current datetime: {datetime_now_cl.strftime("%Y-%m-%d %H:%M:%S")}')

            # 5. Restart last accumulator volumen if day change
            if datetime_file_cl.date() != datetime_now_cl.date():
                volume_data = {
                    "volumen": 0,
                    "last_accumulator_volume": [int(total_volume), timestamp, datetime_now_cl.strftime("%Y-%m-%d %H:%M:%S") ], # [volumen:int, timestamp:int, datetime_cl:str]
                    "timestamp": timestamp,
                    "datetime_cl": datetime_now_cl.strftime("%Y-%m-%d %H:%M:%S")
                }
                self.write_file(file_path, json.dumps(volume_data), "w")

            # 6. Update acum total volumen (total_volume)
            accumulator_day_volume = int(total_volume - volume_data["last_accumulator_volume"][0])
            volume_data["volumen"] = accumulator_day_volume if accumulator_day_volume >= 0 else 0
            volume_data["timestamp"] = timestamp
            volume_data["datetime_cl"] = datetime_now_cl.strftime("%Y-%m-%d %H:%M:%S")
            self.write_file(file_path, json.dumps(volume_data), "w")

        except Exception as ex:
            print(ex)
               
    def create_base_volume_data(self, file_path):
        try:
            default_dict_data = {
                "volumen": 0,
                "last_accumulator_volume": [0, 0, "2000-1-1 00:00:00"], # [volumen:int, timestamp del utimo reinicio del acumulador:int, datetime_cl:str]
                "timestamp": 0,  # Timestamp de la ultima vez que se actualizo el archivo
                "datetime_cl": "2000-1-1 00:00:00" # Datime CL de la ultima vez que se actualizo el archivo
            }
            self.write_file(file_path, json.dumps(default_dict_data), "w")

        except Exception as ex:
            print(f"Error creando archivo base volumen_acumulado.json: {ex} ")

