import sys
import json
import os
import time

sys.path.append('/srv/datalogger_mmr/')
from lib.utils import Utils

SOURCE_CONFIG = "/srv/datalogger_mmr/config_mmr.json"
SOURCE_GEOFENCE = "/srv/datalogger_mmr/geofence.json"

#TODO: ADD AUTOUPDATE GEOFENCE
class Config(Utils):
	def __init__(self, log_id = "CONFIG"):
		self.log_id = log_id
		self.machine_id = self.get_product_id()
		self.machine_name = self.get_product_name()
		self.location = self.get_location_assigned()
		self.faena = self.get_faena_assigned()
		self.avalaible_faenas = self.get_avalaible_faenas()
	
	def check_exist(self, source_config):
		self.log("Checking config files...")

		# --- 1) SI CONFIG NO EXISTE, CREAR COMPLETO ---
		if not os.path.isfile(source_config):
			self.log("Config_mmr.json not found")
			config_mmr = {
				"SERVER":{
					"IP": "127.0.0.1",
					"PORT": 20001,
					"MAX_LEN_PACKET_DATA": 60,
					"SECONDS_MICRODATA": 1,
				},
				"GPS_SENSOR": {
					"PORT": "GPS",
					"MODEL": "GPS_USB",
				},
				"AUTOUPLOAD": {
					"MINING":{
						"ACTIVATED": 1,
						"DATA": "",
						"LOCATION": "",
					},
				},
				"VALVE_SENSOR":{
					"PORT": 5,
					"LEVEL_CURVE": [0, 0],  # Lista con min and max voltage signal: [v_min, v_max]
				},
				"FLOW_SENSOR":{
					"PORT": 4,
					"MODBUS_ID": [3, 4]
				},
				"GEOFENCE_URL": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/geofence' if self.faena in self.avalaible_faenas else "",
				"TEST_MODE":0,
				"TABLET":{
					"MAX_CISTERN_LEVEL": 70,	# Max nivel de estanque del aljibe: default 70m3
					"MODEL_SENSOR_LEVEL": "", #TUF - SATEL - ""
					"SHOW_LEVEL": False,
					"SHOW_VOLUME": False,
					"API_URL_REG": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/vehicles/reg' if self.faena in self.avalaible_faenas else "https://candelaria.mapa.mine-360.com/api/vehicles/reg",
					"API_PM100_FILTRADO": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/tablets/pm100_filtrado' if self.faena in self.avalaible_faenas else "https://candelaria.mapa.mine-360.com/api/tablets/pm100_filtrado",
					"API_RIEGO": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/riego' if self.faena in self.avalaible_faenas else "https://candelaria.mapa.mine-360.com/api/riego",
					"GEOFENCE_PATH": "/srv/datalogger_mmr/geofence.json",
					"TITLE": self.faena,
					"X_COORD":   self.get_x_coord() if self.faena in self.avalaible_faenas else -27.519101,
        			"Y_COORD": self.get_y_coord() if self.faena in self.avalaible_faenas else -70.289366
				},
				"SENSORS_ENABLED":{
        		"flow_left": True,
        		"flow_right": True,
        		"ltr": True,
        		"ltc": True,
        		"rtr": True,
        		"rtc": True
				}
			}
			
			with open(source_config, "w") as f:
				json.dump(config_mmr, f, indent=len(config_mmr))
				
			self.log("Config_mmr.json created")
			return
		
		# --- 2) SI CONFIG EXISTE, VALIDAR KEYS ---
		self.log("Config_mmr.json exists. Validating content...")

		with open(source_config, "r") as f:
			config = json.load(f)

		# Asegurar que "TABLET" exista
		if "TABLET" not in config:
			self.log("TABLET section not found. Adding default TABLET block...")
			config["TABLET"] = {}
			
		# Estructura base del bloque TABLET
		default_tablet = {
			"MAX_CISTERN_LEVEL": 70,
			"MODEL_SENSOR_LEVEL": "",
			"SHOW_LEVEL": False,
			"SHOW_VOLUME": False,
			"API_URL_REG": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/vehicles/reg'
                       if self.faena in self.avalaible_faenas else "",
			"API_PM100_FILTRADO": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/tablets/pm100_filtrado'
                             if self.faena in self.avalaible_faenas else "",
			"API_RIEGO": f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/riego'
                    if self.faena in self.avalaible_faenas else "",
			"GEOFENCE_PATH": "/srv/datalogger_mmr/geofence.json",
			"TITLE": self.faena,
			"X_COORD": self.get_x_coord(),
        	"Y_COORD": self.get_y_coord()		
		}

		# Asegurar que "SENSORS_ENABLED" exista
		if "SENSORS_ENABLED" not in config:
			self.log("SENSORS_ENABLED section not found. Adding default SENSORS block...")
			config["SENSORS_ENABLED"] = {}

		default_sensors = {
        		"flow_left": True,
        		"flow_right": True,
        		"ltr": True,
        		"ltc": True,
        		"rtr": True,
        		"rtc": True
    			}

		# Revisar key por key y agregar si falta
		updated = False
		for key, default_value in default_tablet.items():
			if key not in config["TABLET"]:
				self.log(f"Adding missing TABLET key: {key}")
				config["TABLET"][key] = default_value
				updated = True
		
		for key, default_value in default_sensors.items():
			if key not in config["SENSORS_ENABLED"]:
				self.log(f"Adding missing SENSOR key: {key}")
				config["SENSORS_ENABLED"][key] = default_value
				updated = True

		# Si hubo cambios → guardar archivo
		if updated:
			with open(source_config, "w") as f:
				json.dump(config, f, indent=4)
			self.log("TABLET section updated with missing keys.")
		else:
			self.log("TABLET block OK. No updates required.")


	def get_x_coord(self):
		"""Devuelve la coordenada X según la faena asignada."""
		
		faena_key = self.faena.replace(" ", "").lower()

		FAENA_X = {
			"ministrohales": -22.38916,
			"centinela": -23.003623528779894,
			"candelaria": -27.519101
		}

		return FAENA_X.get(faena_key, 0)	

	def get_y_coord(self):
		"""Devuelve la coordenada Y según la faena asignada."""

		faena_key = self.faena.replace(" ", "").lower()

		FAENA_Y = {
			"ministrohales": -68.916165,
			"centinela": -69.08605917310669,
			"candelaria": -70.289366
		}

		return FAENA_Y.get(faena_key, 0)



	def update_mining_config(self, source_config):
		print("UPDATE MINING CONFIG")
		TOPIC_DATA= f"{self.faena}/dataloggers/{self.location}/{self.machine_name}"
		TOPIC_LOCATION = f"{self.faena}/dataloggers/mapa/{self.location}/{self.machine_name}"
		GEOFENCE_URL = f'https://{self.faena.replace(" ", "").lower()}.mapa.mine-360.com/api/geofence' if self.faena in self.avalaible_faenas else "" 
		with open(SOURCE_CONFIG, 'r') as f:
			config = json.load(f)
		
		if config["AUTOUPLOAD"]["MINING"]["DATA"] != TOPIC_DATA or config["AUTOUPLOAD"]["MINING"]["LOCATION"] != TOPIC_LOCATION or config["GEOFENCE_URL"] != GEOFENCE_URL:
			config["AUTOUPLOAD"]["MINING"]["DATA"] = TOPIC_DATA
			config["AUTOUPLOAD"]["MINING"]["LOCATION"] = TOPIC_LOCATION
			config["GEOFENCE_URL"] = GEOFENCE_URL

			with open(source_config, "w") as f:
				json.dump(config, f, indent=len(config))

			db_restart_cmd = "sudo python3 /srv/datalogger_mmr/database/models.py --reset_db true"
			self.command(db_restart_cmd)
			print("CONFIGURACION MINING ACTUALIZADA")
		
			# Update autoupload services
			self.restart_service("mining-autoupload")
			self.restart_service("mining-autoupload-gps")
		else:
			print("CONFIGURACION MINING YA ESTA ACTUALIZADA")


	def update_geofence(self, source_config, source_geofence):
		try:
			print("Check Geofence")
			if self.faena in self.avalaible_faenas:
			# 1. Check if geofence files exist
				if not os.path.isfile(source_geofence):
					print("Geofence.json not found")
					# Download geofence from API
					self.download_geofence(source_config, source_geofence)
					self.restart_service("mining-gps")

				else:
					print("Geofence.json found")
					# checking for updates
					print("Checking for updates")
					last_timestamp_geofence = os.path.getmtime(source_geofence)
					if time.time() - last_timestamp_geofence >= 60*60*24*7: # 7 dias
						print("Geofence.json is older than 7 days, try update")
						# Update geofence from API
						self.download_geofence(source_config, source_geofence)
						self.restart_service("mining-gps")
					else: print("Geofence not change in the last 7 days")

					# Check is geofence file is broken
					try:
						with open(source_geofence) as file:
							data = json.load(file)
					except:
						self.traceback()
						print("Geofence file is broken, downloading again ... ")
						self.download_geofence(source_config, source_geofence)
						self.restart_service("mining-gps")
			else:
				print("No hay faena asignada")
		except:
			self.traceback()
			self.log("Error updating geofence")

	def download_geofence(self, source_config, source_geofence):
		try:
			config =json.load(open(source_config))
			api_url = config["GEOFENCE_URL"]
			if api_url:
				source_raw_geofence = "/srv/datalogger_mmr/raw_geofence.json"
				geofence_cmd = f"wget {api_url} -O {source_raw_geofence}"

				print(f"Downloading GEOFENCE geometry from: {api_url}")
				print("[%s] ..." % geofence_cmd)
				self.command(geofence_cmd)

				try:
					raw_geofence =json.load(open(source_raw_geofence))
					if "ok" in raw_geofence:
						print("Geofence tiene la nueva estructura, cambiando ...")
						# Cambiar la estructura
						geofence = raw_geofence["data"][0]["row_to_json"]
						with open(source_geofence, "w") as f:
							json.dump(geofence, f)
						print("Geofence guardada con estructura corregida")

					else:
						print("Estructura geofence correcta!")
						geofence = raw_geofence
						with open(source_geofence, "w") as f:
							json.dump(geofence, f)
						print("Geofence guardada")
					
				except:
					print("Error procesando geofence")

				# update tablet geofence
				print("Updating tablet geofence...")
				tablet_cmd = f"cp {source_geofence} /var/www/html/caminos.geojson"
				self.command(tablet_cmd)

				print("update_geofence(): Done")
			else:
				print("No existe url para descarga de geofence")
		except:
			self.traceback()
			self.log("Error updating geofence")

	def check_tablet(self):
		try:
			print("Check Tablet")
			if self.faena in self.avalaible_faenas:
				print("Check if index.html Tablet change...")
				path_current_index = "/var/www/html/index.html"
				faena = self.faena.replace(" ", "")
				path_base_index = f"/srv/datalogger_mmr/tablet/htmls/{faena}.html"
				size_current_index = os.path.getsize(path_current_index)
				size_base_index = os.path.getsize(path_base_index)
				if size_current_index != size_base_index:
					print("Index Tablet change --> Actualizando .......")
					command = f"cp {path_base_index} {path_current_index}"
					self.command(orden = command)
				else:
					print("Index Tablet has not changed")
			else:
				print("No hay faena asignada")
		
		except:
			self.traceback()
			self.log("Error update tablet files")
		
if __name__ == "__main__":
	config = Config()
	print("Actualizando configuraciones")
	# 1. Check if config files exist
	config.check_exist(source_config= SOURCE_CONFIG)

	# 2. Update mining config
	config.update_mining_config(source_config= SOURCE_CONFIG)

	# 3. Update geofence
	config.update_geofence(source_config= SOURCE_CONFIG, source_geofence= SOURCE_GEOFENCE)

	# 4. Check Tablet
	config.check_tablet()
	
