import json
import math
import serial
import socket
import threading
from time import sleep, time
from datetime import datetime
import sys

sys.path.append('/srv/datalogger_mmr/')
from lib.utils import Utils
from database.models import Database


class Server(Utils):
    def __init__(self, max_len_packet_data, seconds_microdata, ip, port, mode, log_id = "SERVER"):

        # Class variables
        self.log_id = log_id
        self.mode = mode
        self.packet_data = []
        self.max_len_packet_data = max_len_packet_data
        self.seconds_microdata = seconds_microdata
        self.timer_microdata = serial.serialutil.Timeout(self.seconds_microdata)

        # Auxiliary variables, flags and acummulators
        self.flag_flow = False
        self.flag_gps = False
        self.count = 0
        self.acum_speed = []
        self.acum_sq = {"left": [], "right":[]}
        self.last_volume = {"left": None, "right":None}
        self.last_flow = {"left": None, "right": None, "timestamp": None}
        self.prev_volume = {"left": None, "right": None}
        self.last_flow_state = None
        self.acum_valve_state = [None, None, None, None]
        self.last_flow_data = {}
        self.last_valve_state = {}
        self.last_gps_data = {}
        self.coords = {"Antucoya": {"Cachimba": {"lat": -22.629696, "lon": -69.869091}},
                       "Candelaria": {"Cachimba_TSHOP": {"lat": -27.524824, "lon": -70.303575}, "Cachimba_F12": {"lat": -27.521344, "lon": -70.303299}, "Cachimba_FINO": {"lat": -27.503510, "lon": -70.282393}},
                       "Centinela": {"Cachimba_SUR": {"lat": -23.016257, "lon": -69.099335}, "Cachimba_TP12": {"lat": -22.999684, "lon": -69.088435}}}
        self.sensors_enabled = self.load_sensor_flags()
        self.last_delta = {"l": 0, "r": 0}
        self.previous_right_volume = None
        self.previous_left_volume = None
        self.previous_flow_time = None

        # Server variables
        self.local_ip = ip
        self.local_port = port
        self.buffer_size = 1024
        self.UDPServerSocket = socket.socket(family=socket.AF_INET, type=socket.SOCK_DGRAM)
        self.UDPServerSocket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.UDPServerSocket.bind((self.local_ip, self.local_port))
        
        
        self.reset_microdata()
        self.database = Database()
        threading.Thread(target=self.read_client_data).start()

    def read_client_data(self)-> None:
        """_summary_
        """
        self.log(f"Server init listening [%s port %s]" % (self.local_ip, self.local_port))
        while True:
            try:
                # Wait from msg clients
                bytesAddressPair = self.UDPServerSocket.recvfrom(self.buffer_size)
                message = bytesAddressPair[0].decode('utf-8')
                msg =  json.loads(message)
                #print(msg)
                name_id = msg['name_id']
                self.process_data(name_id, msg)

                # Wait for flow or valve states
                if self.timer_microdata.expired():
                    microdata = self.construct_microdata()
                    if self.flag_flow:
                        self.packet_data.append(microdata)
                        self.log(f"dato numero {self.count} : {microdata}")
                        self.count = self.count +1 if (self.count is not None) else 1
                        if "valve_state" in microdata:
                            for i, key in enumerate(["ltr","ltc","rtr","rtc"]):
                                if self.sensors_enabled.get(key, False):
                                    if microdata["valve_state"][i] not in (-1, None):
                                        if self.acum_valve_state[i] is not None:
                                            self.acum_valve_state[i] += microdata["valve_state"][i]
                                        else:
                                            self.acum_valve_state[i] = microdata["valve_state"][i]

                                
                    if self.flag_gps:
                        flow_state = 0
                        if not self.last_flow.get("timestamp",0)==0:
                            lat = microdata.get("latitude")
                            lon = microdata.get("longitude")
                            in_cachimba = False

                            if lat is not None and lon is not None: 
                                in_cachimba = self.is_in_coords(lat, lon, radius=10)
                                if not in_cachimba:
                                    if self.last_flow["timestamp"] != self.previous_flow_time:
                                        f_l_state = self.sensors_enabled.get("flow_left", False)
                                        f_r_state = self.sensors_enabled.get("flow_right", False)
                                        left_flow = self.last_flow["left"] if f_l_state and self.last_flow["left"] is not None else 0
                                        right_flow = self.last_flow["right"] if f_r_state and self.last_flow["right"] is not None else 0
                                        #DEFAULT VALUE 1.0
                                        if left_flow > 1 or right_flow > 1:
                                            flow_state = 1
                        
                        dict_data = {
                            "latitude": microdata["latitude"],
                            "longitude": microdata["longitude"],
                            "speed": microdata["speed_kmh"],
                            "zone_id": microdata["geofence_id"],
                            "timestamp": microdata["timestamp"],
                            "valve_state": 1 if 1 in microdata.get("valve_state", [0,0,0,0]) else 0, # TODO: aca considerar si el flujometro esta marcando dato
                            "flow_state": flow_state

                        }
                        
                        with open("/data_live.json", "w") as f:
                            f.write(json.dumps(dict_data))

                    # Restart variables and timer
                    self.reset_microdata()
                    self.timer_microdata.restart(self.seconds_microdata)
                    self.previous_flow_time = self.last_flow["timestamp"]

                # Wait until complete n packet data
                if self.count >= self.max_len_packet_data:
                    if self.packet_data:
                        self.log("Packet data")
                        avg_speed = self.calculate_average(self.acum_speed)
                        mean_sq_left = self.calculate_average(self.acum_sq["left"])
                        mean_sq_right = self.calculate_average(self.acum_sq["right"])
                        prev_vl, prev_vr = self.database.get_last_volume()
                        delta_vl = 0
                        delta_vr = 0
                        
                        lat = microdata.get("latitude")
                        lon = microdata.get("longitude")
                        in_cachimba = False

                        if lat is not None and lon is not None: 
                            in_cachimba = self.is_in_coords(lat, lon, radius=10)

                        if not in_cachimba:
                            if prev_vl is not None and self.last_volume["left"] is not None and self.last_flow["left"] > 0: 
                                delta_vl = self.last_volume["left"] - prev_vl

                            if prev_vr is not None and self.last_volume["right"] is not None and self.last_flow["right"] > 0:
                                delta_vr = self.last_volume["right"] - prev_vr

                        if not self.mode: #No esta en modo test, por lo que debe guardar datos
                            self.acum_valve_state = [0 if value is None else value for value in self.acum_valve_state]
                            self.database.insert_mmr_data(self.packet_data, avg_speed, self.acum_valve_state, self.last_volume["left"], self.last_volume["right"], delta_vl, delta_vr, mean_sq_left, mean_sq_right)
                        else:
                            self.log("Cuidado equipo en modo test, recuerde pasar a modo normal despues de pruebas")

                    # Restart Variables
                    self.packet_data = []
                    self.count = 0
                    self.acum_speed = []
                    self.acum_sq = {"left": [], "right":[]}
                    self.last_volume = {"left": None, "right":None}
                    self.last_flow = {"left": None, "right": None, "timestamp":self.previous_flow_time}
                    self.acum_valve_state = [None, None, None, None]
                    self.prev_volume = {"left": None, "right": None}
                    if self.last_volume["left"] is not None:
                        self.previous_left_volume = self.last_volume["left"]
                    if self.last_volume["right"] is not None:
                        self.previous_right_volume = self.last_volume["right"]
                sleep(0.01)

            except Exception as ex:
                self.traceback()

    def reset_microdata(self)-> None:
        # Reset all dicts
        self.last_flow_data = {}
        self.last_valve_state = {}
        self.last_gps_data = {}
        self.flag_flow = False
        self.flag_gps = False
    
    def construct_microdata(self) -> dict:
        microdata = {}
        microdata.update(self.last_flow_data)
        microdata.update(self.last_valve_state)
        microdata.update(self.last_gps_data)
        microdata["timestamp"] = int(time()*1000)
        microdata["valve_state"] = [self.last_valve_state["m_ltr"], self.last_valve_state["m_ltc"], self.last_valve_state["m_rtr"], self.last_valve_state["m_rtc"]]


        v_l = self.last_flow_data.get("v_l")
        v_r = self.last_flow_data.get("v_r")
        f_l = self.last_flow_data.get("f_l")
        f_r = self.last_flow_data.get("f_r")

        if not self.sensors_enabled.get("flow_left", False):
            microdata["delta_vl"] = -1
        else: 
            microdata["delta_vl"] = self.compute_delta(v_l, f_l, "left", microdata)
        
        if not self.sensors_enabled.get("flow_right", False):
            microdata["delta_vr"] = -1
        else: 
            microdata["delta_vr"] = self.compute_delta(v_r, f_r, "right", microdata)
        
        if self.sensors_enabled.get("flow_left", False) and v_l is not None:
            self.prev_volume["left"] = v_l
        if self.sensors_enabled.get("flow_right", False) and v_r is not None: 
            self.prev_volume["right"] = v_r

        return microdata


    def process_data(self, name_id: str, payload: dict)-> None:
        """_summary_

        Args:
            name_id (str): _description_
            payload (dict): _description_
        """
        self.sensors_enabled = self.load_sensor_flags()

        if name_id == "SERIAL":
            if any([self.sensors_enabled.get(x, False) for x in ["ltr","ltc","rtr","rtc"]]):
                self.flag_flow = True
                if self.acum_valve_state == [None, None, None, None]:
                    self.acum_valve_state = [0 if self.sensors_enabled.get(s, False) else None 
                                            for s in ["ltr","ltc","rtr","rtc"]]
                for i, key in enumerate(["ltr","ltc","rtr","rtc"]):
                    if self.sensors_enabled.get(key, False):
                        self.last_valve_state["m_" + key] = payload["data"].get("m_" + key, -1)
                    else:
                        self.last_valve_state["m_" + key] = -1

        elif name_id == "GPS_USB":
            self.flag_gps = True
            self.acum_speed.append(payload["data"]["speed_kmh"])
            self.last_gps_data.update(payload["data"])

        elif name_id == "FLOW":
            self.flag_flow = True

            if self.sensors_enabled.get("flow_left", False):
                self.last_volume["left"] = payload["data"].get("v_l")
                self.last_flow["left"] = payload["data"].get("f_l")
                self.acum_sq["left"].append(payload["data"].get("sq_l"))
                v_l = payload["data"].get("v_l")
                f_l = payload["data"].get("f_l")
                sq_l = payload["data"].get("sq_l")
            else:
                self.last_volume["left"] = self.previous_left_volume
                self.last_flow["left"] = None
                v_l = None
                f_l = None
                sq_l = None

            if self.sensors_enabled.get("flow_right", False):
                self.last_volume["right"] = payload["data"].get("v_r")
                self.last_flow["right"] = payload["data"].get("f_r")
                self.acum_sq["right"].append(payload["data"].get("sq_r"))
                v_r = payload["data"].get("v_r")
                f_r = payload["data"].get("f_r")
                sq_r = payload["data"].get("sq_r")
            else:
                self.last_volume["right"] = self.previous_right_volume
                self.last_flow["right"] = None
                v_r = None
                f_r = None
                sq_r = None

            timestamp = payload["data"].get("timestamp")
            self.last_flow["timestamp"]=timestamp

            safe_flow_data = {
                "v_l": v_l,
                "f_l": f_l,
                "sq_l": sq_l,
                "v_r": v_r,
                "f_r": f_r,
                "sq_r": sq_r,
                "timestamp": timestamp
            }
            self.last_flow_data.update(safe_flow_data)

        else:
            print(f"That's not a valid type: {name_id}")
            
    def load_sensor_flags(self):
        cfg = json.load(open("/srv/datalogger_mmr/config_mmr.json"))
        return cfg.get("SENSORS_ENABLED", {})

    def compute_delta(self, v, f , side, microdata):
        if not self.sensors_enabled.get(f"flow_{side}", False):
            return -1
        
        if self.last_flow.get("timestamp", 0) == 0:
            return 0
        
        lat = microdata.get("latitude")
        lon = microdata.get("longitude")

        if lat and lon and self.is_in_coords(lat, lon, radius=10):
            return 0
        
        prev = self.prev_volume.get(side)

        if prev is None: 
            return 0
        
        if f is None or f<=0:
            return 0 
        
        delta = v - prev

        self.last_delta[side] = delta

        return delta

    def calculate_average(self, data):
        """Calcula el promedio de una lista de datos."""
        filter_list = [item for item in data if item is not None] # Filter None data
        return round(sum(filter_list) / len(filter_list), 2) if filter_list else None    
    
    def distance(self, lat1, lon1, lat2, lon2):
        lat1, lon1, lat2, lon2 = map(float, (lat1, lon1, lat2, lon2))
        r = 6371000
        d_lat = math.radians(lat2 - lat1)
        d_lon = math.radians(lon2 - lon1)
        a = (math.sin(d_lat/2)**2 + math.cos(math.radians(lat1)) * math.cos(math.radians(lat2)) * math.sin(d_lon/2)**2)
        c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
        return r*c
    
    def is_in_coords(self, lat, lon, radius = 50):
        for __ , location in self.coords.items():
            for __ , point in location.items():
                dist = self.distance(lat, lon, point["lat"], point["lon"])
                if dist <= radius:
                    return True
        return False
        

# Config
f = open('/srv/datalogger_mmr/config_mmr.json')
config:dict = json.load(f)

if __name__ == "__main__":
	Server( max_len_packet_data= config["SERVER"]["MAX_LEN_PACKET_DATA"], 
            seconds_microdata= config["SERVER"]["SECONDS_MICRODATA"],
            ip=config["SERVER"]["IP"], 
            port=config["SERVER"]["PORT"],
            mode = config["TEST_MODE"]
            )
