import threading
from serial import Serial
from time import time, sleep
import json
import sys

sys.path.append('/srv/datalogger_tablet/nrf/')
from lib.usb_dev_node import USBDevnode
from lib.utils import Utils

class SerialLib(Utils):
    def __init__(self, usbdevnode:USBDevnode, data_to_send:str, baudrate: int = 9600) -> None:
        self.log_id = "SERIAL NRF"
        self.timeout = 0.5
        self.baudrate = baudrate
        self.usbdevnode = usbdevnode
        self.data_to_send = data_to_send
        self.seconds_send = 2 # seconds
        self.last_send_timestamp = time()
        self.usb_port = self.usbdevnode.get_devnode()
        threading.Thread(target = self.connect).start()
        self.log("mining nrf initialized")

    def connect(self) -> None:
        """
        This function attempts to establish a serial connection with the specified USB device node.
        """

        try:
            print(f"Try to connect serial port: {self.usb_port}")
            self.serial_module = Serial(self.usb_port, self.baudrate, timeout=self.timeout)
            self.send_data()

        except Exception as Ex:
            print(Ex)

    def send_data(self) -> None:
        """
        This function continuously send data to arduino
        """

        print("Sending data to arduino")
        while True:
            try:
                if self.serial_module.is_open:
                    self.serial_module.write(bytes(self.data_to_send, 'utf-8'))
                    self.last_send_timestamp = time()
                    print(f"Data send: {self.data_to_send}")
                    sleep(self.seconds_send)
                else:
                    print("Serial port is not open. Try to reconnect in 10 seconds...")
                    self.connect()
                    sleep(10)

            except Exception as ex:
                print(ex)
            sleep(0.01)


# Global variables
f = open('/srv/datalogger_tablet/nrf/config.json')
config:dict = json.load(f)
DATA_TO_SEND = config["machine"]
PORT = "nano"

if __name__ == "__main__":
    devnode = USBDevnode(PORT)
    TX = SerialLib(devnode, data_to_send=DATA_TO_SEND)
    TX.set_panic_command("systemctl restart mining-nrf")
    print("mining nrf initialized")

    while True:
        # Breathe
        silence_period = time() - TX.last_send_timestamp
        if int(silence_period) > 60:
            print("Too much TX silence")
        print("last_send_timestamp: %s" % str(silence_period))
        sleep(1)



