from flask import Flask, request, jsonify, render_template, Response, jsonify
from flask_cors import CORS
import json, os, uuid
import time  
from filelock import FileLock, Timeout

app = Flask(__name__, static_folder='static')
CORS(app)

# JSON_PATH = 'button_state.json' ##para dev
# BOTONES_JSON_PATH = 'botonesActivos.json' ##para dev
# MSG_NANO_PATH = 'msg_nano.json' ## Para dev
# NETWORK_PATH = 'network.json' ## Para dev
# GLOBAL_COUNT_PATH = 'global_count.json' ## Para dev

LOCK_TIMEOUT = 5   # segundos antes de asumir que el lock está colgado
RETRY_DELAY = 0.2  # segundos entre intentos


JSON_PATH = '/srv/datalogger_tablet/server_cachimba/Antucoya_html/button_state.json'
BOTONES_JSON_PATH = '/srv/datalogger_tablet/server_cachimba/Antucoya_html/botonesActivos.json'
MSG_NANO_PATH = '/srv/datalogger_tablet/server_cachimba/Antucoya_html/msg_nano.json'
NETWORK_PATH = '/srv/datalogger_tablet/server_cachimba/Antucoya_html/network.json'
GLOBAL_COUNT_PATH = '/srv/datalogger_tablet/server_cachimba/Antucoya_html/global_count.json'

# Estado inicial de acciones y botones
estado_inicial = {}
botones_inicial = {}
msg_nano_inicial = {}

def cargar_json(path, inicial, timeout=5):
    if not os.path.exists(path):
        with open(path, "w") as f:
            json.dump(inicial, f)

    lock = FileLock(path + ".lock")
    try:
        with lock.acquire(timeout=timeout):
            with open(path, "r") as f:
                return json.load(f)
    except Timeout:
        print(f"[ERROR] Timeout al leer {path}")
        return inicial
    except Exception as e:
        print(f"[ERROR] Fallo al leer {path}: {e}")
        return inicial

def guardar_json(path, data, timeout=5):
    lock = FileLock(path + ".lock")
    try:
        with lock.acquire(timeout=timeout):
            with open(path, "w") as f:
                json.dump(data, f, indent=2)
    except Timeout:
        print(f"[ERROR] Timeout al escribir {path}")
    except Exception as e:
        print(f"[ERROR] Fallo al escribir {path}: {e}")


@app.route('/')
def index():
    return render_template('Antucoya.html')

@app.route('/accion', methods=['POST'])
def actualizar_estado():
    data = request.get_json()
    if not data or not isinstance(data, dict):
        return jsonify({"error": "JSON inválido"}), 400

    estado = cargar_json(JSON_PATH, estado_inicial)
    now = int(time.time())

    # Preparar nuevos valores tomando el estado actual si no se envía en data
    nuevo_estado = {
        "cargar_agua": data.get("cargar_agua", estado.get("cargar_agua", False)),
        "cargar_sulf": data.get("cargar_sulf", estado.get("cargar_sulf", False)),
        "detener": data.get("detener", estado.get("detener", False)),
    }

    # Si se pide detener, todo lo demás debe apagarse
    if nuevo_estado["detener"]:
        nuevo_estado["cargar_agua"] = False
        nuevo_estado["cargar_sulf"] = False

    # Validar exclusión mutua
    if nuevo_estado["cargar_agua"] and nuevo_estado["cargar_sulf"]:
        return jsonify({"error": "No se puede activar agua y surfactante al mismo tiempo."}), 400

    estado.update(nuevo_estado)
    estado["time"] = now

    guardar_json(JSON_PATH, estado)
    return jsonify({"status": "ok", "nuevo_estado": estado})




@app.route('/estadoAguaSurf', methods=['GET']) ## Estado de cargar agua o surf
def obtener_estado():
    estado = cargar_json(JSON_PATH, estado_inicial)
    return jsonify(estado)

@app.route('/estadoBotonesActivos', methods=['GET'])  # Estado de botones Activos
def estado_botones():

    try:
        # Leer network.json
        with open(NETWORK_PATH, 'r') as f:
            net = json.load(f)
        id_mmr = net.get('id_mmr', None)
        network = net.get('network', True)

        # Leer global count json
        try:
            with open(GLOBAL_COUNT_PATH, 'r') as file_count:
                global_count = json.load(file_count)
        except: 
            global_count = {"activar_camara": 0}
            with open(GLOBAL_COUNT_PATH, "w") as file_count:
                json.dump(global_count, file_count, indent=4)
        
        # Leer msg_nano.json
        msg_nano = cargar_json(MSG_NANO_PATH, msg_nano_inicial)
        
        # Activar cámara solo si la clave con el nombre del id_mmr está en True
        activar_camara = True
        if id_mmr and msg_nano.get(id_mmr, False) is True:
            activar_camara = True
            global_count["activar_camara"] = 0
        else: 
            global_count["activar_camara"]=global_count["activar_camara"]+1
        
        if global_count["activar_camara"]>5:
            activar_camara = False
            with open(GLOBAL_COUNT_PATH, "w") as file_count:
                json.dump(global_count, file_count, indent=4)
        else:
            activar_camara = True
            with open(GLOBAL_COUNT_PATH, "w") as file_count:
                json.dump(global_count, file_count, indent=4)
        
        #if not network:
        #    print("[INFO] Red no disponible, asumiendo que no hay red.")
        #    activar_camara = False

        botones_inicial['activar_camara'] = activar_camara
        print(f"[INFO] Activar camara: {botones_inicial['activar_camara']} (id_mmr: {id_mmr})")

    except FileNotFoundError:
        print("[ERROR] Archivo de red no encontrado, asumiendo que no hay red.")
        botones_inicial['activar_camara'] = True  # Fallback

    return jsonify(botones_inicial)


@app.route('/id_mmr', methods=['GET'])
def obtener_id_mmr():
    try:
        with open(NETWORK_PATH, 'r') as f:
            net = json.load(f)
        id_mmr = net.get('id_mmr', '')
    except FileNotFoundError:
        id_mmr = ''

    resp = Response(id_mmr, mimetype='text/plain; charset=utf-8')
    resp.headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0'
    return resp


@app.route('/modo', methods=['GET'])
def modo_operacion():
    try:
        #resp = requests.get('', timeout=1) llamado al mqtt
        #data = resp.json()
        data = cargar_json(MSG_NANO_PATH , msg_nano_inicial) # Simulación de respuesta de la Jetson
        modo = data.get("modo_operacion")
        print(f"[INFO] modo carga recibido: {data}")
        return jsonify({"modo_operacion": modo})
    except Exception as e:
        print(f"[ERROR] al consultar la Jetson: {e}")
        return jsonify({"modo_operacion": 1})


#### IA
@app.route('/msg_nano')

def consultar_json():
    data = cargar_json(MSG_NANO_PATH , msg_nano_inicial) 

    print(f"[INFO] Datos Jetson: {data}")
    try:
        pass
    except:
        pass
    return jsonify(data) 



if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000)
