setInterval(() => {
  console.log("identificador:", identificador);
}, 1000); // 1000 ms = 1 segundo

// ====== Config ======
const REG_URL = "https://antucoya.mapa.mine-360.com/api/vehicles/reg";
const RIEGO_BASE = "https://antucoya.mapa.mine-360.com/api/grafico/regadores";
const PM100_BASE = "https://antucoya.mapa.mine-360.com/api/pm100data";
const TZ_CHILE   = "America/Santiago"; 
const RIEGO_FRESH_SECONDS = 3600; // 1 hora; ajusta a gusto
const PM100_RADIUS_M = 200; // registros a <= 200 m del regador se consideran "en su ubicación"


// ====== Utilidades ======
function fetchWithTimeout(resource, { timeout = 8000, ...options } = {}) {
  const controller = new AbortController();
  const id = setTimeout(() => controller.abort(), timeout);
  return fetch(resource, { ...options, signal: controller.signal }).finally(
    () => clearTimeout(id)
  );
}


/* ===== Haversine para distancia en metros ===== */
function haversineMeters(lat1, lon1, lat2, lon2) {
  const R = 6371000, toRad = x => x * Math.PI / 180;
  const dLat = toRad(lat2 - lat1), dLon = toRad(lon2 - lon1);
  const a = Math.sin(dLat/2)**2 + Math.cos(toRad(lat1))*Math.cos(toRad(lat2))*Math.sin(dLon/2)**2;
  return 2 * R * Math.asin(Math.sqrt(a));
}

// Edad (seg) entre PM (UTC) y fecha de referencia (vehículo en SCL correctamente parseado)
function ageSecondsRelative(pmItem, refDate) {
  if (!pmItem?.time_utc) return Infinity;
  const pmDt  = new Date(pmItem.time_utc.replace(" ", "T") + "Z"); // parseUtc
  const refDt = refDate instanceof Date ? refDate : parseVehicleTimeInSantiago(refDate);
  if (!pmDt || !refDt || isNaN(pmDt) || isNaN(refDt)) return Infinity;
  return Math.abs(pmDt.getTime() - refDt.getTime()) / 1000;
}


function fmtM3(x) {
  const n = Number(x);
  if (!isFinite(n)) return "—";
  return n.toLocaleString("en-US", { maximumFractionDigits: 2 });
}

function parseLocalChileToDate(naive) {
  if (!naive) return null;
  const s = String(naive).replace("_", " ");
  const [d, t="00:00"] = s.split(" ");
  const [y,m,day] = d.split("-").map(Number);
  const [hh,mi]   = t.split(":").map(Number);
  // Probar offsets CLT (-04) y CLST (-03) y elegir el que, formateado en SCL, calce el string
  const candidates = [-4,-3].map(off => new Date(Date.UTC(y, m-1, day, hh - off, mi, 0)));
  const target = `${d} ${String(hh).padStart(2,"0")}:${String(mi).padStart(2,"0")}:00`;
  for(const dt of candidates){
    if (formatInTZ(dt, TZ_CHILE) === target) return dt;
  }
  // Fallback CLT
  return new Date(Date.UTC(y, m-1, day, hh + 4, mi, 0));
}

// Extrae el número de "REG-002", "REG-02", "REG-2", "reg_02", etc.
function regNumber(str) {
  if (!str) return null;
  const m = String(str).match(/reg[-_ ]*0*(\d+)/i);
  return m ? parseInt(m[1], 10) : null;
}

function numOrNull(x) {
  const n = Number(x);
  return Number.isFinite(n) ? n : null;
}

function isTimestampKey(k) {
  // Acepta "YYYY-MM-DD HH:mm:ss" o con 'T'
  return /^\d{4}-\d{2}-\d{2}[ T]\d{2}:\d{2}:\d{2}$/.test(String(k));
}

// Busca dentro del objeto { "REG-01": {...}, "REG-02": {...} } usando el número
function pickRegFromGrafico(data, identificador) {
  target = regNumber(identificador);
  if (target == null || !data) return null;
  const key = Object.keys(data).find((k) => regNumber(k) === target);
  return key ? { key, payload: data[key] } : null;
}

function todayStr(tz = TZ_CHILE) {
  const parts = new Intl.DateTimeFormat("en-CA", {
    timeZone: tz,
    year: "numeric",
    month: "2-digit",
    day: "2-digit",
  }).formatToParts(new Date());
  const y = parts.find((p) => p.type === "year").value;
  const m = parts.find((p) => p.type === "month").value;
  const d = parts.find((p) => p.type === "day").value;
  return `${y}-${m}-${d}`; // YYYY-MM-DD
}

// Parse "YYYY-MM-DD HH:mm:ss" como UTC -> Date
function parseUtc(tsUtc) {
  if (!tsUtc) return null;
  // "2025-08-19 04:00:00" -> "2025-08-19T04:00:00Z"
  return new Date(tsUtc.replace(" ", "T") + "Z");
}

// Devuelve SOLO el objeto del vehículo que coincide con tu identificador
function getVehicleByIdentificador(lista, identificador) {
  target = regNumber(identificador);
  console.log("target:", target);
  if (target == null) return null;
  return lista.find((v) => regNumber(v.vehiculo) === target) || null;
}

function formatTimeAgo(dt) {
  if (!(dt instanceof Date)) return "—";
  const s = Math.floor((Date.now() - dt.getTime()) / 1000);
  if (s < 0) return "—";
  if (s < 60) return `hace ${s}s`;
  const m = Math.floor(s / 60);
  if (m < 60) return `hace ${m}m`;
  const h = Math.floor(m / 60);
  return `hace ${h}h`;
}

function formatInTZ(date, tz = "America/Santiago") {
  if (!(date instanceof Date) || isNaN(date)) return "—";
  const p = new Intl.DateTimeFormat("en-CA", {
    timeZone: tz, year:"numeric", month:"2-digit", day:"2-digit",
    hour:"2-digit", minute:"2-digit", second:"2-digit", hour12:false
  }).formatToParts(date);
  const get = t => p.find(x => x.type === t)?.value ?? "00";
  return `${get("year")}-${get("month")}-${get("day")} ${get("hour")}:${get("minute")}:${get("second")}`;
}

// Convierte un string naive "YYYY-MM-DDTHH:mm:ss" que está en hora Chile → Date UTC correcto
function parseVehicleTimeInSantiago(naive) {
  if (!naive) return null;
  const s = String(naive).replace("T", " ");
  const [d, t="00:00:00"] = s.split(" ");
  const [y, m, day] = d.split("-").map(Number);
  const [hh, mi, ss=0] = t.split(":").map(Number);

  // Probar offsets posibles de Chile para esa fecha: -04 (CLT) y -03 (CLST)
  const candidates = [-4, -3].map(off => {
    // “Hora Chile” → UTC: sumamos el offset (e.g., 00:00 CLT → 04:00 UTC)
    const ms = Date.UTC(y, m-1, day, hh - off, mi, ss);
    return new Date(ms);
  });

  // Elegimos el candidato cuyo formateo en SCL calce exactamente con la hora naive
  const target = `${String(y).padStart(4,"0")}-${String(m).padStart(2,"0")}-${String(day).padStart(2,"0")} ${String(hh).padStart(2,"0")}:${String(mi).padStart(2,"0")}:${String(ss).padStart(2,"0")}`;
  for (const dte of candidates) {
    if (formatInTZ(dte, "America/Santiago") === target) return dte;
  }
  // Fallback razonable: CLT (-04)
  return new Date(Date.UTC(y, m-1, day, hh + 4, mi, ss));
}

// Trae datos de riego y deja listo acumulado y última hora
async function getRiegoForIdentificador() {
  const url = `${RIEGO_BASE}/${todayStr(TZ_CHILE)}`;
  const res = await fetchWithTimeout(url, { headers: { Accept: "application/json" }, timeout: 8000 });
  if (!res.ok) throw new Error(`HTTP ${res.status} al consultar ${url}`);
  const data = await res.json();

  const picked = pickRegFromGrafico(data, identificador);
  if (!picked) return null;

  const { payload } = picked; // { m3_hora: [{hora, m3}, ...], acumulado_total }
  const acumulado = Number(payload.acumulado_total || 0);

  let horaHHmm = "—";
  let m3Hora = null;
  let dtSCL = null;

  if (Array.isArray(payload.m3_hora) && payload.m3_hora.length > 0) {
    // Orden por hora local y último
    const sorted = [...payload.m3_hora].sort((a,b) => String(a.hora).localeCompare(String(b.hora)));
    const last = sorted[sorted.length - 1]; // {hora: 'YYYY-MM-DD_HH:mm', m3: n}
    const horaRaw = last?.hora ?? null;
    dtSCL = horaRaw ? parseLocalChileToDate(horaRaw) : null;

    // solo "HH:mm" para mostrar
    const show = horaRaw ? horaRaw.replace("_"," ").split(" ")[1] : null; // "HH:mm"
    horaHHmm = show || "—";
    m3Hora = Number(last?.m3 ?? 0);
  }

  // Frescura del acumulado según el último registro horario
  const ageSec = dtSCL ? (Date.now() - dtSCL.getTime())/1000 : Infinity;
  const isFresh = ageSec <= RIEGO_FRESH_SECONDS;

  return { acumulado, horaHHmm, m3Hora, dtSCL, ageSec, isFresh };
}

/* ===== Fetch PM100 del día ===== */

// Elige SIEMPRE el más cercano; la “frescura” se mide vs ultimo_dato del regador
function chooseNearestPm100WithRef(pmList, regLat, regLon, vehUltimoDato, syncSec = 600) {
  if (!Array.isArray(pmList) || !Number.isFinite(regLat) || !Number.isFinite(regLon)) return null;

  const candidates = pmList.filter(p =>
    Number.isFinite(p.latitude) && Number.isFinite(p.longitude) && p.pm100 != null
  );
  if (!candidates.length) return null;

  // Más cercano en espacio
  let best = null, bestD = Infinity;
  for (const p of candidates) {
    const d = haversineMeters(regLat, regLon, p.latitude, p.longitude);
    if (d < bestD) { bestD = d; best = p; }
  }
  if (!best) return null;

  // Frescura relativa al regador (ahora sí, con `ultimo_dato` bien interpretado en SCL)
  const vehDate = vehUltimoDato instanceof Date ? vehUltimoDato : parseVehicleTimeInSantiago(vehUltimoDato);
  const ageRelSec = ageSecondsRelative(best, vehDate);
  const isFresh   = ageRelSec <= syncSec;

  // Logs de verificación en SCL
  const pmDt = new Date(best.time_utc.replace(" ", "T") + "Z");
  console.log("[DIAG] Veh  SCL:", formatInTZ(vehDate, "America/Santiago"));
  console.log("[DIAG] PM   SCL:", formatInTZ(pmDt,   "America/Santiago"));
  console.log("[DIAG] Δt(s):", Math.round(ageRelSec), "fresh:", isFresh);

  return { ...best, distance_m: bestD, ageSec: ageRelSec, isFresh };
}

// Devuelve el registro más reciente (por timestamp_utc) dentro de un radio alrededor del REG.
// Si no hay nada en el radio, devuelve null (o, si prefieres, el más cercano fuera de radio como fallback).
function pickLatestPm100Near(pmList, regLat, regLon, radiusM = PM100_RADIUS_M) {
  if (!Array.isArray(pmList) || !Number.isFinite(regLat) || !Number.isFinite(regLon)) return null;

  // 1) Filtrar a los que tienen coords/pm100 y están dentro del radio
  const inRadius = [];
  for (const p of pmList) {
    if (!Number.isFinite(p.latitude) || !Number.isFinite(p.longitude) || p.pm100 == null) continue;
    const d = haversineMeters(regLat, regLon, p.latitude, p.longitude);
    if (d <= radiusM) {
      inRadius.push({ ...p, distance_m: d });
    }
  }

  if (inRadius.length === 0) {
    console.warn(`[PM100] No hay registros dentro de ${radiusM} m de la ubicación del REG.`);
    const timeEl= document.getElementById("hud-polvo-time");
    if (timeEl) timeEl.textContent = "No hay registro cerca";
    return null; // o: return nearest fuera de radio si quieres fallback
  }

  // 2) Ordenar por tiempo (UTC) y tomar el último
  inRadius.sort((a, b) => {
    const da = parseUtc(a.time_utc)?.getTime() ?? -Infinity;
    const db = parseUtc(b.time_utc)?.getTime() ?? -Infinity;
    return da - db;
  });

  const latest = inRadius[inRadius.length - 1];
  // Para mostrar bonita la hora local SCL
  latest._time_scl = formatInTZ(parseUtc(latest.time_utc), "America/Santiago");

  console.log("[PM100] Elegido (último en ubicación REG):", latest);
  return latest;
}



// Trae /api/pm100data/YYYY-MM-DD y transforma a array ordenado por hora local
// === Función con logs detallados ===
async function getPm100Data(dateStr = todayStr()) {
  const url = `${PM100_BASE}/${dateStr}`;
  console.log(`[PM100] Fetch -> ${url}`);

  const res = await fetchWithTimeout(url, { headers: { Accept: "application/json" }, timeout: 8000 });
  console.log(`[PM100] HTTP status: ${res.status}, ok=${res.ok}`);
  if (!res.ok) throw new Error(`HTTP ${res.status} al consultar ${url}`);

  const raw = await res.clone().text();
  console.log("[PM100] Raw (primeros 300 chars):", raw.slice(0, 300));

  let obj;
  try {
    obj = JSON.parse(raw);
  } catch (e) {
    console.error("[PM100] ERROR parseando JSON:", e);
    return [];
  }

  if (!obj || typeof obj !== "object") {
    console.warn("[PM100] Top-level no es objeto:", obj);
    return [];
  }
  console.log("[PM100] Top-level keys:", Object.keys(obj));

  // Detecta payload (por si viene como {data: {...}} o {status:..., data: {...}})
  let payload = obj;
  if (obj && typeof obj === "object" && "data" in obj) {
    payload = obj.data;
    console.log("[PM100] Usando obj.data como payload");
  } else if (obj && typeof obj === "object" && "result" in obj) {
    payload = obj.result;
    console.log("[PM100] Usando obj.result como payload");
  }

  // Normaliza a array de registros {time_local, time_utc, lat, lon, pm100}
  let arr = [];

  if (Array.isArray(payload)) {
    console.log("[PM100] Payload es array. length:", payload.length);
    arr = payload.map((v, i) => {
      const time_local = v?.time_local || v?.timestamp_local || v?.time || v?.timestamp || null;
      const time_utc   = v?.timestamp_utc ?? null;
      return {
        time_local,
        time_utc,
        latitude:  numOrNull(v?.latitude),
        longitude: numOrNull(v?.longitude),
        pm100:     numOrNull(v?.pm100),
      };
    });
  } else if (payload && typeof payload === "object") {
    const keys = Object.keys(payload);
    console.log("[PM100] Payload es objeto. keys:", keys.slice(0, 100), `... total=${keys.length}`);

    // Filtra solo las claves con formato de timestamp
    const tsEntries = Object.entries(payload).filter(([k]) => isTimestampKey(k));
    console.log(`[PM100] Claves con timestamp: ${tsEntries.length} / ${keys.length}`);

    // Si no encontró claves de tiempo, loguea para diagnóstico
    if (tsEntries.length === 0) {
      console.warn("[PM100] No hay claves con timestamp. Ejemplos de keys:", keys.slice(0, 10));
    }

    arr = tsEntries.map(([time_local, v]) => ({
      time_local,                                   // "YYYY-MM-DD HH:mm:ss"
      time_utc: v?.timestamp_utc ?? null,           // "YYYY-MM-DD HH:mm:ss" (UTC)
      latitude:  numOrNull(v?.latitude),
      longitude: numOrNull(v?.longitude),
      pm100:     numOrNull(v?.pm100),
    }));
  } else {
    console.warn("[PM100] Payload vacío o tipo desconocido:", payload);
    return [];
  }

  // Orden por tiempo local si existe
  arr.sort((a, b) => String(a.time_local).localeCompare(String(b.time_local)));

  // Métricas de calidad
  const withCoords = arr.filter(r => r.latitude != null && r.longitude != null);
  const withPm     = arr.filter(r => r.pm100 != null);
  const full       = arr.filter(r => r.latitude != null && r.longitude != null && r.pm100 != null);

  console.log(`[PM100] Registros totales: ${arr.length}`);
  console.log(`[PM100] Con coords: ${withCoords.length}`);
  console.log(`[PM100] Con pm100: ${withPm.length}`);
  console.log(`[PM100] Con coords + pm100: ${full.length}`);
  console.log("[PM100] Ejemplo 1er registro:", arr[0]);
  console.log("[PM100] Ejemplo último registro:", arr[arr.length - 1]);

  return arr;
}


/* ===== Buscar el punto PM100 más cercano a (lat, lon) ===== */
function nearestPm100(pmList, lat, lon) {
  if (!Array.isArray(pmList) || pmList.length === 0) return null;
  let best = null, bestD = Infinity;
  for (const p of pmList) {
    const d = haversineMeters(lat, lon, p.latitude, p.longitude);
    if (d < bestD) { bestD = d; best = p; }
  }
  return best ? { ...best, distance_m: bestD } : null;
}

// ====== HUD  ======
const elSpeed = document.getElementById("hud-speed");
const elStamp = document.getElementById("hud-stamp");
const elDot = document.getElementById("hud-dot");

// ====== Actualización HUD → Velocidad ======
function setHUDNoData() {
  if (elSpeed) elSpeed.textContent = "—";
  if (elStamp) elStamp.textContent = "Sin datos";
  if (elDot) elDot.classList.remove("fresh");
}

function updateHUD(veh) {
  if (!veh) {
    setHUDNoData();
    return;
  }
  const v =
    typeof veh.speed_kmh === "number"
      ? veh.speed_kmh
      : Number(veh.speed_kmh) || 0;
  if (elSpeed) elSpeed.textContent = Math.round(v);

  // Marca "fresco" si el último dato es reciente (<= 20 s); ajusta a gusto
  const dt =
    veh.ultimo_dato instanceof Date
      ? veh.ultimo_dato
      : new Date(veh.ultimo_dato);
  const ageSec = (Date.now() - dt.getTime()) / 1000;
  if (elDot) {
    if (ageSec <= 20) elDot.classList.add("fresh");
    else elDot.classList.remove("fresh");
  }
  if (elStamp) elStamp.textContent = `Actualizado ${formatTimeAgo(dt)}`;
}

// ====== Actualización HUD → Riego ======
const elRiego = document.getElementById("hud-riego");
const elRiegoBadge = document.querySelector(
  "#hud .hud-card:nth-child(3) .hud-badge"
); // tercera tarjeta (Riego)

function setHUDRiegoNoData() {
  if (elRiego) elRiego.textContent = "—";
  if (elRiegoBadge) elRiegoBadge.textContent = "sin datos";
}

function updateHUDRiego(info){
  const acumEl = document.getElementById("hud-riego-acum");
  const dot    = document.getElementById("hud-riego-dot");
  const lastT  = document.getElementById("hud-riego-last-time");
  const lastV  = document.getElementById("hud-riego-last-value");

  if (!info){
    if (acumEl) acumEl.textContent = "—";
    if (dot)    dot.classList.remove("fresh");
    if (lastT)  lastT.textContent = "—";
    if (lastV)  lastV.textContent = "—";
    return;
  }

  const { acumulado, horaHHmm, m3Hora, ageSec, isFresh } = info;

  if (acumEl) acumEl.textContent = isFinite(acumulado) ? Number(acumulado).toLocaleString('es-CL', { maximumFractionDigits: 2 }) : "—";
  if (dot)    dot.classList.toggle("fresh", !!isFresh);
  if (lastT)  lastT.textContent = horaHHmm || "—";
  if (lastV)  lastV.textContent = (m3Hora != null ? Number(m3Hora).toLocaleString('es-CL', { maximumFractionDigits: 2 }) : "—");
}


/* ===== Actualización HUD → Polvo ===== */
const elPolvo = document.getElementById("hud-polvo");
const elPolvoBadge = document.querySelector("#hud .hud-card:nth-child(2) .hud-badge"); // tarjeta 2 = Polvo

function setHUDPolvoNoData() {
  const vEl   = document.getElementById("hud-polvo-value");
  const dot   = document.getElementById("hud-polvo-dot");
  const stamp = document.getElementById("hud-polvo-stamp");
  const distEl= document.getElementById("hud-polvo-dist");
  const timeEl= document.getElementById("hud-polvo-time");

  if (vEl)   vEl.textContent = "—";
  if (dot)   dot.classList.remove("fresh");
  if (stamp) stamp.textContent = "Sin datos";
  if (distEl)distEl.textContent = "—";
  if (timeEl)timeEl.textContent = "—";
}

function updateHUDPolvo(info){
  const vEl   = document.getElementById("hud-polvo-value");
  const dot   = document.getElementById("hud-polvo-dot");   // lo usamos como indicador "en radio" (verde) o no (gris)
  const distEl= document.getElementById("hud-polvo-dist");
  const timeEl= document.getElementById("hud-polvo-time");

  if(!info){
    if (vEl)   vEl.textContent = "—";
    if (dot)   dot.classList.remove("fresh");
    if (distEl)distEl.textContent = "—";
    if (timeEl)timeEl.textContent = "—";
    return;
  }

  // Valor
  const val = Number.isFinite(info.pm100) ? info.pm100 : null;
  if (vEl) vEl.textContent = val == null ? "—" : val.toLocaleString('es-CL');

  // Indicador: verde si está dentro del radio configurado
  if (dot) dot.classList.toggle("fresh", Number.isFinite(info.distance_m) && info.distance_m <= PM100_RADIUS_M);

  // Sello: “Último registro (ubicación REG)”
  const tsTxt = info._time_scl || (info.time_utc ? formatInTZ(parseUtc(info.time_utc), "America/Santiago") : "");

  // Distancia + hora (SCL)
  if (distEl) distEl.textContent = Number.isFinite(info.distance_m) ? `${Math.round(info.distance_m)} metros` : "—";
  if (timeEl) timeEl.textContent = tsTxt.split(" ")[1] || tsTxt; // muestra "HH:mm:ss"
  
}




// ====== Polling ======
async function getRegVehicles() {
  const res = await fetchWithTimeout(REG_URL, {
    headers: { Accept: "application/json" },
    timeout: 8000,
  });
  if (!res.ok) throw new Error(`HTTP ${res.status} al consultar ${REG_URL}`);
  const data = await res.json();
  console.log("Vehículos registrados:", data);
  return data.map((v) => ({ ...v, ultimo_dato: new Date(v.ultimo_dato) }));
}

async function pollHUD() {
  try {
    const regsPromise  = getRegVehicles?.() ?? Promise.resolve(null);
    const riegoPromise = getRiegoForIdentificador?.() ?? Promise.resolve(null);
    const pmPromise    = getPm100Data(todayStr(TZ_CHILE));

    const [regs, riegoInfo, pmList] = await Promise.all([regsPromise, riegoPromise, pmPromise]);

    // Velocidad
    let mine = null;
    if (Array.isArray(regs) && typeof getVehicleByIdentificador === "function") {
      mine = getVehicleByIdentificador(regs, identificador);
      if (typeof updateHUD === "function") updateHUD(mine);
    }

    
    // Polvo (punto más cercano; frescura relativa al regador)
// Polvo: último registro en la ubicación del REG
if (mine && Number.isFinite(mine.latitude) && Number.isFinite(mine.longitude)) {
  const chosen = pickLatestPm100Near(pmList, mine.latitude, mine.longitude, PM100_RADIUS_M);
  updateHUDPolvo(chosen);
} else {
  setHUDPolvoNoData();
}




    // Riego
    if (typeof updateHUDRiego === "function") updateHUDRiego(riegoInfo);

  } catch (err) {
    console.error("[HUD] Error:", err);
    setHUDPolvoNoData();
    if (typeof setHUDRiegoNoData === "function") setHUDRiegoNoData();
    if (typeof setHUDNoData === "function") setHUDNoData();
  }
}

// Intervalo
pollHUD();
window.__hudInterval && clearInterval(window.__hudInterval);
window.__hudInterval = setInterval(pollHUD, 15000);
