import paho.mqtt.client as mqtt
import json
import os
import tempfile
import time
import logging
from datetime import datetime

# Configuración remota
broker = "mqtt.produccion.mine-360.com"
port = 1883
username = "SkyMine"
password = "SkyMining2025*"
topic = "estado/general"
json_output_path = "/srv/datalogger_tablet/server_cachimba/Antucoya_html/msg_nano.json"
log_path = "/srv/datalogger_tablet/server_cachimba/local_client_mqtt_data/msg_nano.log"
fallback_timeout = 5  # segundos sin mensaje antes de fallback

# Configurar logger
os.makedirs(os.path.dirname(log_path), exist_ok=True)
logging.basicConfig(
    filename=log_path,
    level=logging.WARNING,
    format="%(asctime)s [%(levelname)s] %(message)s",
)

ultimo_mensaje_recibido = time.time()

# Fallback si no hay mensaje
def fallback_si_ausente():
    global ultimo_mensaje_recibido
    if time.time() - ultimo_mensaje_recibido > fallback_timeout:
        data_fallback = {
            "modo_operacion": 1,
            "centrado": None,
            "estado": None,
            "numeroCamion": None,
            "videoSrc": None,
            "error": "No se recibieron mensajes MQTT",
            "timestamp": datetime.utcnow().isoformat() + "Z"
        }

        try:
            os.makedirs(os.path.dirname(json_output_path), exist_ok=True)
            with tempfile.NamedTemporaryFile("w", delete=False, dir=os.path.dirname(json_output_path)) as tmpfile:
                json.dump(data_fallback, tmpfile, indent=2)
                temp_path = tmpfile.name
            os.replace(temp_path, json_output_path)
            logging.warning("No se recibieron mensajes MQTT. Se escribió fallback en msg_nano.json.")
        except Exception as e:
            logging.error("Error escribiendo fallback JSON: %s", e)

# Callback de mensaje recibido
def on_message(client, userdata, msg):
    global ultimo_mensaje_recibido
    print("Mensaje recibido del tópico:", msg.topic)
    try:
        payload = msg.payload.decode("utf-8")
        data = json.loads(payload)
    except Exception as e:
        logging.error("Error al decodificar o cargar JSON: %s", e)
        return

    base = {
        "modo_operacion": 1,
        "centrado": None,
        "estado": None,
        "numeroCamion": None,
        "videoSrc": None,
        "error": None,
        "timestamp": None
    }

    data["modo_operacion"] = 1
    base.update(data)

    try:
        os.makedirs(os.path.dirname(json_output_path), exist_ok=True)
        with tempfile.NamedTemporaryFile("w", delete=False, dir=os.path.dirname(json_output_path)) as tmpfile:
            json.dump(base, tmpfile, indent=2)
            temp_path = tmpfile.name
        os.replace(temp_path, json_output_path)
        print("JSON guardado correctamente en", json_output_path)
        ultimo_mensaje_recibido = time.time()
    except Exception as e:
        logging.error("Error escribiendo el archivo JSON: %s", e)

# Callback al conectar
def on_connect(client, userdata, flags, rc):
    if rc == 0:
        print(f"Conectado a {broker}:{port}")
        client.subscribe(topic)
        print("Suscrito al tópico:", topic)
    else:
        logging.error(f"Fallo al conectar al broker (código {rc})")

# Callback al desconectar
def on_disconnect(client, userdata, rc):
    if rc != 0:
        logging.warning("Desconexión inesperada del broker MQTT.")

# Crear cliente MQTT
client = mqtt.Client(protocol=mqtt.MQTTv311)
client.username_pw_set(username=username, password=password)
client.on_connect = on_connect
client.on_message = on_message
client.on_disconnect = on_disconnect

# Intentar conexión persistente
print("Conectando al broker MQTT remoto...")
while True:
    try:
        client.connect(broker, port, 60)
        print("Conectado al broker", broker)
        break
    except Exception as e:
        logging.warning("No se pudo conectar al broker MQTT: %s", e)
        print("Reintentando conexión en 5 segundos...")
        time.sleep(5)

client.loop_start()

# Loop para fallback
try:
    while True:
        fallback_si_ausente()
        time.sleep(1)
except KeyboardInterrupt:
    print("Finalizando cliente MQTT...")
finally:
    client.loop_stop()
    client.disconnect()
