import paho.mqtt.client as mqtt
import json
import os
import tempfile
import time
import logging
from datetime import datetime

# Configuracion
broker = "nvidia-desktop.local"
port = 1884
topic = "estado/general"
json_output_path = "/srv/datalogger_tablet/server_cachimba/Antucoya_html/msg_nano.json"
log_path = "/srv/datalogger_tablet/server_cachimba/local_client_mqtt_data/msg_nano.log"
fallback_timeout = 5  # segundos sin mensaje antes de fallback

# Configurar logger
os.makedirs(os.path.dirname(log_path), exist_ok=True)
logging.basicConfig(
    filename=log_path,
    level=logging.WARNING,
    format="%(asctime)s [%(levelname)s] %(message)s",
)

# Ultimo tiempo en que se recibio un mensaje MQTT
ultimo_mensaje_recibido = time.time()

# Fallback para guardar JSON por defecto si no se reciben mensajes
def fallback_si_ausente():
    global ultimo_mensaje_recibido
    if time.time() - ultimo_mensaje_recibido > fallback_timeout:
        data_fallback = {
            "modo_operacion": 1,
            "centrado": None,
            "estado": None,
            "numeroCamion": None,
            "videoSrc": None,
            "error": "No se recibieron mensajes MQTT",
            "timestamp": datetime.utcnow().isoformat() + "Z"
        }

        try:
            os.makedirs(os.path.dirname(json_output_path), exist_ok=True)
            with tempfile.NamedTemporaryFile("w", delete=False, dir=os.path.dirname(json_output_path)) as tmpfile:
                json.dump(data_fallback, tmpfile, indent=2)
                temp_path = tmpfile.name
            os.replace(temp_path, json_output_path)
            logging.warning("No se recibieron mensajes MQTT. Se escribio fallback en msg_nano.json.")
        except Exception as e:
            logging.error("Error escribiendo fallback JSON: %s", e)

# Callback al recibir mensaje
def on_message(client, userdata, msg):
    global ultimo_mensaje_recibido
    print("Mensaje recibido del topico:", msg.topic)
    try:
        payload = msg.payload.decode("utf-8")
        data = json.loads(payload)
    except Exception as e:
        logging.error("Error al decodificar o cargar JSON: %s", e)
        return

    # Rellenar con valores por defecto si faltan campos
    base = {
        "modo_operacion": 1,
        "centrado": None,
        "estado": None,
        "numeroCamion": None,
        "videoSrc": None,
        "error": None,
        "timestamp": None
    }

    # Forzar modo_operacion a 1
    data["modo_operacion"] = 1
    base.update(data)

    try:
        os.makedirs(os.path.dirname(json_output_path), exist_ok=True)
        with tempfile.NamedTemporaryFile("w", delete=False, dir=os.path.dirname(json_output_path)) as tmpfile:
            json.dump(base, tmpfile, indent=2)
            temp_path = tmpfile.name
        os.replace(temp_path, json_output_path)
        print("JSON guardado correctamente en", json_output_path)
        ultimo_mensaje_recibido = time.time()
    except Exception as e:
        logging.error("Error escribiendo el archivo JSON: %s", e)

# Callback al conectar
def on_connect(client, userdata, flags, rc):
    if rc == 0:
        print("Conectado al broker MQTT:", f"{broker}:{port}")
        client.subscribe(topic)
        print("Suscrito al topico:", topic)
    else:
        logging.error(f"Fallo al conectar al broker (codigo {rc})")

# Callback al desconectar
def on_disconnect(client, userdata, rc):
    if rc != 0:
        logging.warning("Desconexion inesperada del broker MQTT.")

# Crear cliente MQTT
client = mqtt.Client(protocol=mqtt.MQTTv311)
client.on_connect = on_connect
client.on_message = on_message
client.on_disconnect = on_disconnect

# Intentar conexion de forma persistente
print("Conectando al broker MQTT...")
while True:
    try:
        client.connect(broker, port, 60)
        print("Conectado al broker", broker)
        break
    except Exception as e:
        logging.warning("No se pudo conectar al broker MQTT: %s", e)
        print("Reintentando conexion en 5 segundos...")
        time.sleep(5)

client.loop_start()

# Loop para fallback
try:
    while True:
        fallback_si_ausente()
        time.sleep(1)
except KeyboardInterrupt:
    print("Finalizando cliente MQTT...")
finally:
    client.loop_stop()
    client.disconnect()
