from flask import Flask, render_template, jsonify, request
from flask_cors import CORS
import json
import os
import time

# ==========================
# CARGA DE CONFIGURACIÓN
# ==========================

#PROD
BASE_DIR = "/srv/datalogger_tablet"
CONFIG_PATH = BASE_DIR + "/config_mmr.json"


# Ruta absoluta del archivo de configuración
#BASE_DIR = "./"
#CONFIG_PATH = Path(__file__).parent / "Config_mmr.json"
#JSON_DIR_REG = "json_dir_reg" 

def load_config_hot():
    """Carga el archivo Config_mmr.json SOLO si cambió en disco."""
    global _config_cache, _config_mtime

    try:
        current_mtime = os.path.getmtime(CONFIG_PATH)

        # Si el archivo NO cambió - regresar cache
        if _config_cache is not None and current_mtime == _config_mtime:
            return _config_cache

        # Si cambió - recargar
        print("🔄 Recargando configuración (Config_mmr.json cambió)")

        with open(CONFIG_PATH, "r", encoding="utf-8") as f:
            _config_cache = json.load(f)

        _config_mtime = current_mtime
        return _config_cache

    except Exception as e:
        print(f"❌ Error cargando configuración: {e}")
        return {}


# Cache interno
_config_cache = None
_config_mtime = 0

# Cargar configuración
config = load_config_hot()

# ==========================
# CONFIGURACIÓN FLASK
# ==========================

app = Flask(__name__, template_folder='templates', static_folder='static')
CORS(app)  # Evita errores CORS desde el navegador

# ==========================
# RUTAS API
# ==========================

@app.route('/config')
def get_config():
    """Devuelve las variables del archivo Config_mmr.json en la key Tablet"""

    cfg = load_config_hot().get("TABLET", {})
    cfg_api = load_config_hot().get("API_TOKEN")
    print(cfg)

    return jsonify({
        "SHOW_LEVEL": cfg.get("SHOW_LEVEL"),
        "SHOW_VOLUME": cfg.get("SHOW_VOLUME"),
        "API_URL_REG": cfg.get("API_URL_REG"),
        "API_PM100_FILTRADO": cfg.get("API_PM100_FILTRADO"),
        "API_RIEGO": cfg.get("API_RIEGO"),
        "X_COORD": cfg.get("X_COORD"),
        "Y_COORD": cfg.get("Y_COORD"),
        "GEOFENCE_PATH": cfg.get("GEOFENCE_PATH"),
        "TITLE": cfg.get("TITLE"),
        "API_TOKEN": cfg_api
    })

@app.route('/nivel')
def get_nivel():
    """Devuelve los datos del archivo nivel.json"""
    file_path = os.path.join(BASE_DIR, 'nivel.json')
    print(f"📁 Leyendo nivel desde: {file_path}")
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        return jsonify(data)

    except FileNotFoundError:
        return jsonify({'error': 'Archivo nivel.json no encontrado'}), 200
    except json.JSONDecodeError:
        return jsonify({'error': 'Archivo nivel.json inválido'}), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/geofence')
def get_geofence():
    """Devuelve los datos del archivo geofence.json"""

    cfg = load_config_hot().get("TABLET", {})
    geofence_route = cfg.get("GEOFENCE_PATH")

    try:
        with open(geofence_route, "r") as f:
            geofence_data = json.load(f)
            
        features = geofence_data["data"][0]["row_to_json"]["features"]
        
        geojson = {
            "type": "FeatureCollection",
            "features": features
        }
        
        return jsonify(geojson)
    
    except Exception as ex:
        print("Error leyendo geofence:", ex)
        return jsonify({"error": "No se pudo leer geofence"}), 500  

@app.route('/volumen')
def get_volumen():
    """Devuelve los datos del archivo volumen_acumulado.json"""
    file_path = os.path.join(BASE_DIR, 'volumen_acumulado.json')
    print(f"📁 Leyendo volumen desde: {file_path}")
    try:
        # Verificar existencia del archivo
        if not os.path.exists(file_path):
            return jsonify({'error': 'Archivo volumen_acumulado.json no encontrado'}), 404

        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)

        # Validar estructura esperada (opcional)
        if not isinstance(data, dict):
            return jsonify({'error': 'Formato de JSON inválido'}), 400

        return jsonify(data)

    except json.JSONDecodeError:
        return jsonify({'error': 'Archivo volumen_acumulado.json corrupto o mal formado'}), 400
    except Exception as e:
        return jsonify({'error': str(e)}), 500



@app.route('/data')
def get_data():
    """Devuelve los datos del archivo data.json"""
    file_path = os.path.join(BASE_DIR, 'data.json')
    print(f"📁 Leyendo data desde: {file_path}")
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
            print(data)

        return jsonify(data)

    except FileNotFoundError:
        return jsonify({'sin_nivel': True, 'error': 'Archivo data.json no encontrado'}), 200
    except json.JSONDecodeError:
        return jsonify({'sin_nivel': True, 'error': 'Archivo data.json inválido'}), 200
    except Exception as e:
        return jsonify({'error': str(e)}), 500
    

@app.route('/reiniciar_contador', methods=['POST'])
def reiniciar_contador():
    """Recibe un timestamp y actualiza el archivo estado_boton.json"""
    file_path = os.path.join(BASE_DIR, 'estado_boton.json')

    try:
        # Obtener JSON del cliente
        data = request.get_json(force=True)

        # Validar que tenga timestamp
        if "timestamp" not in data:
            return jsonify({"error": "Falta el campo 'timestamp'"}), 400

        # Guardar en el archivo
        with open(file_path, 'w', encoding='utf-8') as f:
            json.dump({"timestamp": time.time()}, f, ensure_ascii=False, indent=2)

        print(f"✅ Contador reiniciado: {data['timestamp']}")
        return jsonify({"status": "ok", "timestamp": data["timestamp"]})

    except json.JSONDecodeError:
        return jsonify({"error": "JSON inválido en la solicitud"}), 400
    except Exception as e:
        return jsonify({"error": str(e)}), 500


# ==========================
# RUTAS HTML
# ==========================

@app.route('/')
def index():
    """Renderiza el HTML principal"""
    return render_template('index.html')


# ==========================
# MAIN
# ==========================

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)
