// mapa.js - inicialización async y modular del mapa Leaflet + lógica eye3

// Variables globales principales
let mymap = null;
window.eye3 = window.eye3 || { configurations: {} };

// ==========================
// Utilidad: esperar(ms)
// ==========================
function esperar(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

// ==========================
// Utilidad: wrapper Promise para $.getJSON
// ==========================
function getJSONAsync(url, params = {}) {
  return new Promise((resolve, reject) => {
    $.getJSON(url, params, function (data) {
      resolve(data);
    }).fail(function (jqxhr, textStatus, error) {
      console.error("Error en getJSONAsync:", textStatus, error);
      reject(error || textStatus);
    });
  });
}

// ==========================
// Animación del panel de info
// ==========================
function initAnimacionPanel() {
  const animacion = function () {
    const panel = document.getElementById("infoPanel");
    if (panel) {
      panel.classList.toggle("fade");
    }
  };
  setInterval(animacion, 500);
}

// ==========================
// Inicialización principal al cargar DOM
// ==========================
document.addEventListener("DOMContentLoaded", () => {
  inicializarApp();
});

async function inicializarApp() {
  try {
    console.log("⏳ Inicializando aplicación eye3...");
    initAnimacionPanel();
    await inicializarMapaAsync();
    inicializarSocketYWatchdog();
    inicializarTimersClusters();
    inicializarTimersMarkers();
    inicializarTimersRiego();
    inicializarTimerDataCenter();
    console.log("✅ Aplicación eye3 inicializada por completo.");
  } catch (err) {
    console.error("❌ Error inicializando la app:", err);
  }
}

// ==========================
// Inicialización async del mapa y capas base
// ==========================
async function inicializarMapaAsync() {
  console.log("⏳ Inicializando mapa Leaflet...");

  const START_ZOOM = 15;
  const MAX_AUTO_ZOOM = 22;

  // Obtiene la long y lat localmente

  const configuracion = await fetch("/config");
  if (!configuracion.ok) throw new Error("No se pudo leer /config");
  const data = await configuracion.json();

  X_COORD = data.X_COORD;
  Y_COORD = data.Y_COORD;

  console.log("Configuración cargada:");
  console.log("X_COORD:", X_COORD);
  console.log("Y_COORD:", Y_COORD);

  // Crear mapa
  mymap = L.map("mapid", {
    zoomControl: false,
    minZoom: 11,
    maxZoom: MAX_AUTO_ZOOM,
  }).setView([X_COORD, Y_COORD], START_ZOOM);

  L.control.zoom({ position: "topright" }).addTo(mymap);

  mymap.on("zoomend moveend", () => console.log("zoom:", mymap.getZoom()));

  $(".leaflet-control-attribution").hide();

  // Estilo día/noche (preservando tu override de hora = 9)
  const fechaActual = new Date();
  let hora = fechaActual.getHours();
  console.log("HORA:", hora);
  hora = 9; // se mantiene tu lógica original

  const MAPBOX_STYLE_DAY = "mapbox/satellite-v9";
  const MAPBOX_STYLE_NIGHT = "mapbox/dark-v11";

  const mapaEstilo =
    hora >= 8 && hora < 20 ? MAPBOX_STYLE_DAY : MAPBOX_STYLE_NIGHT;
  console.log("Estilo del mapa:", mapaEstilo);

  const MAPBOX_TOKEN =
    "pk.eyJ1IjoiZ2plcmV6IiwiYSI6ImNtYWViemQ5bDA1cHMycnB6NnliZnhrdjgifQ.J3NFy110e2pHvO5tN4KA3w";

  L.tileLayer(
    `https://api.mapbox.com/styles/v1/${mapaEstilo}/tiles/{z}/{x}/{y}?access_token=${MAPBOX_TOKEN}`,
    {
      maxZoom: 22,
      tileSize: 512,
      zoomOffset: -1,
      attribution: "© OpenStreetMap · © Mapbox",
      detectRetina: false,
      className: "sat-base",
    }
  ).addTo(mymap);

  console.log("Usando estilo:", mapaEstilo);
  mymap.on("tileerror", (e) => console.error("Tile error:", e));

  // Recalcular tamaño si el contenedor cambia
  setTimeout(() => mymap.invalidateSize(), 300);

  const popup = L.popup(); // reservado si se usa más adelante

  // ==========================
  // Marcadores hardcodeados
  // ==========================
  function addMarker(name, latlon) {
    const marker = new L.Marker(latlon, {
      title: name,
      clickable: false,
      draggable: false,
      icon: L.divIcon({
        className: "",
        iconSize: [100, 40],
        iconAnchor: [50, 40],
        html: `<p align="center" style="align:center;font-family:Helvetica; font-weight:bold; font-size:20px; color:#ffffff; text-shadow: -2px -2px 0 #000, 2px -2px 0 #000, -2px 2px 0 #000, 2px 2px 0 #000, 0px -2px 0 #000, 0px 2px 0 #000, 2px 0px 0 #000, -2px 0px 0 #000;">${name}<p>`,
      }),
    });
    marker.addTo(mymap);
  }
  // Exponer por si se quiere usar desde fuera
  window.addMarker = addMarker;

  // ==========================
  // Objeto global eye3
  // ==========================
  window.eye3 = {
    map_center_update_interval_ms: 5000,
    last_gps_timestamp: 0,
    configurations: {},
    CustomIcon: L.Icon.extend({
      options: {
        iconSize: [50, 50],
        shadowSize: [0, 0],
        iconAnchor: [25, 25],
        shadowAnchor: [4, 0],
        popupAnchor: [0, 0],
      },
    }),
    CustomWaterTruckIcon: L.Icon.extend({
      options: {
        iconSize: [200, 200],
        shadowSize: [0, 0],
        iconAnchor: [34, 120],
        shadowAnchor: [4, 0],
        popupAnchor: [0, 0],
      },
    }),
    HaulTruckIcon: function (angle) {
      const svgicon_url = encodeURI(
        "data:image/svg+xml," + haulTruckSVG(angle)
      ).replace("#", "%23");
      const icon = new eye3.CustomIcon({ iconUrl: svgicon_url });
      return icon;
    },
    WaterTruckIcon: function (name, color) {
      const svgicon_url = encodeURI(ColorSVGMarker(name, color));
      const icon = new eye3.CustomWaterTruckIcon({ iconUrl: svgicon_url });
      return icon;
    },
    map: mymap,
    vehicle: null,
    waterPolylines: [],
    dustMarks: [],
    microClusters: [],
    macroClusters: [],
    macroClusterBounds: [],
    layer: [],
    polygon: [],
    markers: [],
  };

  // ==========================
  // Configuraciones locales
  // ==========================
  const configurations = [
    { name: "position_max_hours", value: "12" },
    { name: "polvo_dia_tablet", value: "90" },
    { name: "polvo_noche_tablet", value: "120" },
    { name: "riego_dia_tablet", value: "30" },
    { name: "riego_noche_tablet", value: "30" },
    { name: "ecom_scale_factor", value: "1" },
    { name: "mapa_inicial_min", value: "1700" },
    { name: "mapa_inicial_max", value: "4000" },
    { name: "rango_pm_alto", value: "3500" },
    { name: "rango_pm_medio_alto", value: "1500" },
    { name: "rango_pm_medio_bajo", value: "600" },
    { name: "rango_pm_bajo", value: "599" },
    { name: "sanidad_switch", value: "1" },
    { name: "horas_muestra_camino", value: "12" },
    { name: "minimo_corte_camino", value: "200" },
    { name: "responsable_geofence_produccion", value: "Evelyn Celis" },
    {
      name: "responsable_geofence_produccion_unificado",
      value: "Evelyn Celis",
    },
    { name: "factorgimppm100", value: "10" },
  ];

  configurations.forEach((configuration) => {
    console.log(configuration.name, configuration.value);
    eye3.configurations[configuration.name] = configuration.value;
  });

  // ==========================
  // Evento click en mapa
  // ==========================
  async function onMapClick(e) {
    const configuracion = await fetch("/config");
    if (!configuracion.ok) throw new Error("No se pudo leer /config");
    const data = await configuracion.json();

    X_COORD = data.X_COORD;
    Y_COORD = data.Y_COORD;

    console.log("Configuración cargada:");
    console.log("X_COORD:", X_COORD);
    console.log("Y_COORD:", Y_COORD);
    const oldlat = X_COORD;
    const oldlon = Y_COORD;
    const newlat = e.latlng.lat;
    const newlon = e.latlng.lng;

    const angle =
      (Math.atan2(newlon - oldlon, newlat - oldlat) * 180) / 3.141592;
    console.log("Angle click:", angle);
  }

  mymap.on("click", onMapClick);

  // ==========================
  // Capas de caminos y riego
  // ==========================
  let colorLayer = hora >= 8 && hora < 20 ? "#000000" : "#f5f6fa";

  eye3.layer["waterLineStrings"] = L.geoJson(null, {
    style: function () {
      return {
        color: "#0000ff",
        weight: 5,
        opacity: 0.5,
        strokeWeight: 1,
      };
    },
  });
  mymap.addLayer(eye3.layer["waterLineStrings"]);

  eye3.layer["caminos"] = L.geoJson(null, {
    style: function () {
      return {
        color: colorLayer,
        fill: true,
        opacity: 0.8,
        clickable: false,
        strokeWeight: 0.5,
      };
    },
    onEachFeature: function (feature, layer) {
      if (feature.properties.polygon_id == null) return;
      eye3.polygon[feature.properties.polygon_id] = layer;
      layer.options.polygon_id = feature.properties.polygon_id;
    },
  });

  try {
    // Obtener directamente el GeoJSON desde el backend
    const resp = await fetch("/geofence");
    if (!resp.ok) throw new Error("No se pudo obtener la geofence");

    const geofenceGeoJSON = await resp.json();

    // Agregar la geofence al mapa
    eye3.layer["caminos"].addData(geofenceGeoJSON);

    console.log("Geofence cargado exitosamente");
  } catch (err) {
    console.error("No se pudo cargar caminos.geojson:", err);
  }
  mymap.addLayer(eye3.layer["caminos"]);

  // ==========================
  // AREAS (construcciones_left)
  // ==========================
  const construcciones_left = L.geoJson(null, {
    style: function (feature) {
      console.log("AREA feature:", feature);
      if (feature.properties.Area === "Caminos") {
        return {
          color: "#4d4d4d",
          strokeColor: "#4d4d4d",
          fillOpacity: 0.7,
          strokeOpacity: 0.3,
          weight: 0.1,
        };
      }
      return {
        color: feature.properties.strokeColor,
        fill: true,
        opacity: 1,
        clickable: false,
      };
    },
  });
  // Nota: en tu código original nunca se agrega al mapa, se respeta eso.

  const d = new Date();
  eye3.lastMapEvent =
    d.getYear() + 1900 + "-" + (d.getMonth() + 1) + "-" + d.getDate();

  eye3.markers = [];

  // Ocultar controles de zoom
  $(".leaflet-control-zoom").hide();
}

// ==========================
// Render de clusters y bounds
// ==========================
function renderMacroClusterBounds(result) {
  for (let i in eye3.macroClusterBounds) {
    eye3.map.removeLayer(eye3.macroClusterBounds[i]);
  }
  eye3.macroClusterBounds = [];
  for (let i in result.geojson_bounds.features) {
    const feature = result.geojson_bounds.features[i];
    const props = feature.properties;
    const coords = feature.geometry.coordinates[0];
    const newcoords = coords.map(function (latlon) {
      return [latlon[1], latlon[0]];
    });

    let boundsColor = "#00ff00";
    if (props.avg_pm100 >= 200 && props.avg_pm100 < 600) {
      boundsColor = "#ff0000";
    }
    if (props.avg_pm100 >= 600 && props.avg_pm100 < 2500) {
      boundsColor = "#990033";
    }
    if (props.avg_pm100 >= 2500 && props.avg_pm100 < 4000) {
      boundsColor = "#990099";
    }

    const p = L.polygon(newcoords, {
      fillColor: boundsColor,
      stroke: false,
      opacity: 0.15,
      fillOpacity: 0.15,
    }).addTo(eye3.map);
    eye3.macroClusterBounds.push(p);
  }
}

function renderMicroClusters(result) {
  for (let i in eye3.microClusters) {
    eye3.map.removeLayer(eye3.microClusters[i]);
  }
  eye3.microClusters = [];
  for (let i in result.geojson_clusterbase.features) {
    const feature = result.geojson_clusterbase.features[i];
    const coords = feature.geometry.coordinates[0];
    let newcoords;
    if (feature.geometry.type === "MultiPolygon") {
      newcoords = coords.map(function (x) {
        return x.map(function (y) {
          return y.map(function (z) {
            return [z[1], z[0]];
          });
        });
      });
    } else {
      newcoords = coords.map(function (latlon) {
        return [latlon[1], latlon[0]];
      });
    }
    console.log(
      `--- procesando cluster ${i} de ${result.geojson_clusterbase.features.length}`
    );
    console.log(newcoords);

    const clusterColor = "#FF0000";

    const p = L.polygon(newcoords, {
      fillColor: clusterColor,
      stroke: false,
      opacity: 0.75,
      fillOpacity: 0.75,
    }).addTo(eye3.map);
    eye3.microClusters.push(p);
  }
}

function renderMacroClusters(result) {
  for (let i in eye3.macroClusters) {
    eye3.map.removeLayer(eye3.macroClusters[i]);
  }
  eye3.macroClusters = [];
  for (let i in result.geojson_clusters.features) {
    const feature = result.geojson_clusters.features[i];
    const coords = feature.geometry.coordinates;
    let newcoords;
    if (feature.geometry.type === "MultiPolygon") {
      newcoords = coords.map(function (x) {
        return x.map(function (y) {
          return y.map(function (z) {
            return [z[1], z[0]];
          });
        });
      });
    } else {
      newcoords = coords.map(function (latlon) {
        return [latlon[1], latlon[0]];
      });
    }

    const clusterColor = "#ff0000";

    newcoords.map(function (c) {
      const polygon_coords = c[0];
      const p = L.polygon(polygon_coords, {
        fillColor: clusterColor,
        stroke: false,
        opacity: 0.65,
        fillOpacity: 0.65,
      }).addTo(eye3.map);
      eye3.macroClusters.push(p);
    });
  }
}

// ==========================
// Cálculo de horario (invierno/verano)
// ==========================
function horaHorario() {
  const fecha = new Date();
  let gmt = 0;
  const year = fecha.getFullYear();
  let mes = parseInt(fecha.getMonth() + 1);
  let dia = fecha.getDate();
  const inicio_invierno = year + "-04-03";
  const termino_invierno = year + "-09-04";

  if (mes < 10) {
    mes = `0${mes}`;
  } else {
    mes = `${mes}`;
  }
  if (dia < 10) {
    dia = `0${dia}`;
  } else {
    dia = `${dia}`;
  }
  const fecha_actual = `${year}-${mes}-${dia}`;

  if (fecha_actual > inicio_invierno && fecha_actual < termino_invierno) {
    console.log("INVIERNO");
    gmt = "GMT-4";
  } else {
    console.log("VERANO");
    gmt = "GMT-3";
  }
  return gmt;
}

// ==========================
// Clusters PM100 (async)
// ==========================
async function updateClusters() {
  if (typeof eye3.waiting_for_clusterdata === "undefined") {
    eye3.waiting_for_clusterdata = false;
  }
  if (eye3.waiting_for_clusterdata === true) {
    return;
  }

  eye3.waiting_for_clusterdata = true;

  try {
    // Obtener configuración desde Flask
    const configRes = await fetch("/config");
    if (!configRes.ok) throw new Error("No se pudo cargar /config");
    const configData = await configRes.json();

    const API_PM100_FILTRADO = configData.API_PM100_FILTRADO;
    console.log("🔗 URL desde Flask (PM100):", API_PM100_FILTRADO);

    $.ajax({url : API_PM100_FILTRADO, method: "GET", dataType: "json", headers: {"X-Device-Key" : configData.API_TOKEN}})
    .done(function (data) {  
    console.log("DATA JSON (highPM100): ", data);
      data = data.data;

      // Limpia las capas anteriores de los clusters
      for (let layer of eye3.macroClusters) {
        eye3.map.removeLayer(layer);
      }
      eye3.macroClusters = [];

      if (!data || data.length === 0) {
        eye3.waiting_for_clusterdata = false;
        return;
      }

      const points = data.map((p) => ({
        lat: p.latitude,
        lng: p.longitude,
        pm100: p.pm100,
        visited: false,
      }));

      const route = [];
      let current = points[0];
      current.visited = true;
      route.push(current);

      while (route.length < points.length) {
        const currentLatLng = L.latLng(current.lat, current.lng);
        let nearestPoint = null;
        let minDistance = Infinity;

        points.forEach((p) => {
          if (!p.visited) {
            const candidateLatLng = L.latLng(p.lat, p.lng);
            const d = currentLatLng.distanceTo(candidateLatLng);
            if (d < minDistance) {
              minDistance = d;
              nearestPoint = p;
            }
          }
        });

        if (nearestPoint) {
          nearestPoint.visited = true;
          route.push(nearestPoint);
          current = nearestPoint;
        } else {
          break;
        }
      }

      const maxSegmentDistance = 200;

      // Crear las líneas (polylines) para los puntos
      for (let i = 0; i < route.length - 1; i++) {
        const startPoint = route[i];
        const endPoint = route[i + 1];
        const startLatLng = L.latLng(startPoint.lat, startPoint.lng);
        const endLatLng = L.latLng(endPoint.lat, endPoint.lng);
        const distance = startLatLng.distanceTo(endLatLng);

        if (distance <= maxSegmentDistance) {
          let color, opacity;
          if (startPoint.pm100 >= 10000) {
            color = "#8B0000";
            opacity = 0.2;
          } else if (startPoint.pm100 >= 1000) {
            color = "#FF6347";
            opacity = 0.1;
          }

          const latlngs = [
            [startPoint.lat, startPoint.lng],
            [endPoint.lat, endPoint.lng],
          ];

          const segment = L.polyline(latlngs, {
            color: color,
            opacity: opacity,
            dashArray: "5, 15",
            lineJoin: "round",
            lineCap: "round",
            weight: 15,
          }).addTo(eye3.map);

          eye3.macroClusters.push(segment);
        }
      }

      eye3.waiting_for_clusterdata = false;
    });
  } catch (err) {
    console.error("Error en updateClusters:", err);
    eye3.waiting_for_clusterdata = false;
  }
}

function inicializarTimersClusters() {
  updateClusters();
  setInterval(function () {
    updateClusters();
  }, 7000);
}

// ==========================
// Marcadores de vehículos (async)
// ==========================
async function updateMarkers() {
  if (typeof eye3.waiting_for_positiondata === "undefined") {
    eye3.waiting_for_positiondata = false;
  }
  if (eye3.waiting_for_positiondata) return;

  eye3.waiting_for_positiondata = true;

  let dataJson = false;
  let regadorid = "";
  let regadorTimestamp = null;

  try {
    const configRes = await fetch("/config");
    if (!configRes.ok) throw new Error("No se pudo cargar /config");
    const configData = await configRes.json();

    const API_URL_REG = configData.API_URL_REG;
    console.log("🔗 URL desde Flask (REG):", API_URL_REG);

    $.getJSON("/data", function (data) {
      dataJson = true;
      console.log(data);
      regadorid = "REG-" + data.name;
      regadorTimestamp = data.timestamp;

      //$.getJSON(API_URL_REG, {}
      $.ajax({url : API_URL_REG, method: "GET", dataType: "json", headers: {"X-Device-Key" : configData.API_TOKEN}})
        .done(function (result) {
        for (let x = 0; x < result.length; x++) {
          const regador = result[x];

          if (regador.latitude == null) {
            regador.latitude = 0;
          }
          if (regador.longitude == null) {
            regador.longitude = 0;
          }

          const splitname = regador.vehiculo.split("-")[1];

          const regadorEndpointTimestamp = regador.ultimo_dato
            ? new Date(regador.ultimo_dato)
            : null;
          const currentTimestamp = regadorEndpointTimestamp
            ? regadorEndpointTimestamp.getTime()
            : null;
          const regadorIsCurrent = regador.vehiculo === regadorid;

          if (regadorIsCurrent && dataJson) {
            const regadorTimestampDate = new Date(regadorTimestamp * 1000);

            if (
              currentTimestamp &&
              regadorTimestamp &&
              currentTimestamp > regadorTimestampDate.getTime()
            ) {
              console.log(
                `Regador: ${regador.vehiculo}, Endpoint Timestamp: ${regador.ultimo_dato}, Data.json Timestamp: ${regadorTimestamp}`
              );
              console.log(
                "Usando datos del endpoint para el regador actual porque el timestamp de data.json es antiguo."
              );
              regadorTimestamp = currentTimestamp;
            } else {
              console.log(
                `Regador: ${regador.vehiculo}, Endpoint Timestamp: ${regador.ultimo_dato}, Data.json Timestamp: ${regadorTimestamp}`
              );
              console.log(
                "Usando datos de data.json para el regador actual porque el timestamp del endpoint es antiguo."
              );
              regador.latitude = data.latitude;
              regador.longitude = data.longitude;
            }
          }

          if (!regadorIsCurrent) {
            if (
              regador.vehiculo != null &&
              regador.latitude != null &&
              regador.longitude != null
            ) {
              if (typeof eye3.markers[regador.vehiculo] === "undefined") {
                eye3.markers[regador.vehiculo] = L.marker([
                  regador.latitude,
                  regador.longitude,
                ]).addTo(eye3.map);
                eye3.markers[regador.vehiculo].setIcon(
                  eye3.WaterTruckIcon(splitname, "rgba(75,71,147,0.8)")
                );
              } else {
                eye3.markers[regador.vehiculo].setIcon(
                  eye3.WaterTruckIcon(splitname, "rgba(75,71,147,0.8)")
                );
                eye3.markers[regador.vehiculo].setLatLng([
                  regador.latitude,
                  regador.longitude,
                ]);
              }
            }
          }
        }
        

        eye3.waiting_for_positiondata = false;})
    .fail(function () {
        console.error("No se pudo obtener los vehículos del endpoint.");
        eye3.waiting_for_positiondata = false;
      })
    .fail(function () {
      dataJson = false;
      regadorid = "";
      console.error("No se pudo obtener /data Llamando a updateMarkers...");
      eye3.waiting_for_positiondata = false;
    });
  })
}
catch (err) {
    console.error("Error en updateMarkers:", err);
    eye3.waiting_for_positiondata = false;
  }
}

function inicializarTimersMarkers() {
  updateMarkers();
  setInterval(function () {
    updateMarkers();
  }, 3000);
}

// ==========================
// Polígonos (updatePolygon)
// ==========================
function updatePolygon(data) {
  const layer = eye3.polygon[data.idpoligono];
  if (typeof layer === "undefined") {
    return;
  }
  eye3.dustMarks.push(layer);

  let texto = "";
  let strokeColor = "";
  let fillColor = "";

  if (data.tipoevento === 0 || data.tipoevento === 1) {
    if (data.tasaideal === 0) {
      texto = "No regar";
      strokeColor = "#00aa00";
      fillColor = strokeColor;
    }
    if (data.tasaideal === 1) {
      texto = "Riego baja concentración";
      strokeColor = "#ff8c00";
      fillColor = strokeColor;
    }
    if (data.tasaideal === 2) {
      texto = "Riego alta concentración";
      strokeColor = "#ff0000";
      fillColor = strokeColor;
    }
  }

  if (data.tipoevento === 2 || data.tipoevento === 3) {
    strokeColor = "#0000ff";
    fillColor = strokeColor;
  }

  console.log("setting polygon " + data.idpoligono + " to " + fillColor);
  layer.setStyle({
    color: strokeColor,
    fillColor: fillColor,
    fillOpacity: 0.4,
  });
}

// ==========================
// Utilidades de segmentos
// ==========================
function areEndpointsClose(latlngs, tolMeters = 30) {
  if (latlngs.length < 2) return false;
  const a = L.latLng(latlngs[0][0], latlngs[0][1]);
  const b = L.latLng(
    latlngs[latlngs.length - 1][0],
    latlngs[latlngs.length - 1][1]
  );
  return a.distanceTo(b) <= tolMeters;
}

function segmentByDistanceAndTime(
  latlngs,
  times = null,
  maxGapMeters = 200,
  maxGapMs = 10 * 60 * 1000
) {
  const segs = [];
  let cur = [];
  for (let i = 0; i < latlngs.length; i++) {
    const pt = latlngs[i];
    if (!cur.length) {
      cur.push(pt);
      continue;
    }
    const prev = cur[cur.length - 1];
    const d = L.latLng(prev[0], prev[1]).distanceTo(L.latLng(pt[0], pt[1]));
    const tOk =
      !times ||
      (i > 0 && Date.parse(times[i]) - Date.parse(times[i - 1]) <= maxGapMs);
    if (d > maxGapMeters || !tOk) {
      if (cur.length > 1) segs.push(cur);
      cur = [pt];
    } else {
      cur.push(pt);
    }
  }
  if (cur.length > 1) segs.push(cur);
  return segs;
}

// ==========================
// Marcadores de riego (async)
// ==========================
async function buscarRiego() {
  if (typeof window.eye3.waiting_for_riego === "undefined") {
    window.eye3.waiting_for_riego = false;
  }
  if (window.eye3.waiting_for_riego) return;
  window.eye3.waiting_for_riego = true;

  try {
    const configRes = await fetch("/config");
    if (!configRes.ok) throw new Error("No se pudo cargar /config");
    const configData = await configRes.json();

    const API_RIEGO = configData.API_RIEGO;
    console.log("🔗 URL desde Flask (RIEGO):", API_RIEGO);

    $.ajax({url : API_RIEGO, method: "GET", dataType: "json", headers: {"X-Device-Key" : configData.API_TOKEN}})
      .done(function (data) {
        eye3.layer["waterLineStrings"].clearLayers();
        if (!Array.isArray(data) || data.length === 0) return;

        const cleaned = data
          .map((d) => ({
            lat: parseFloat(d.latitude),
            lng: parseFloat(d.longitude),
            time: d.time,
          }))
          .filter(
            (d) => Number.isFinite(d.lat) && Number.isFinite(d.lng) && d.time
          );

        cleaned.forEach((d) => {
          L.circleMarker([d.lat, d.lng], {
            radius: 3,
            color: "#0066ff",
            fillOpacity: 0.5,
            opacity: 0.7,
          }).addTo(eye3.layer["waterLineStrings"]);
        });
      })
      .fail(function (jqxhr, textStatus, error) {
        console.error("Error al cargar riego:", textStatus, error);
      })
      .always(function () {
        window.eye3.waiting_for_riego = false;
      });
  } catch (err) {
    console.error("Error en buscarRiego:", err);
    window.eye3.waiting_for_riego = false;
  }
}

function inicializarTimersRiego() {
  buscarRiego();
  setInterval(function () {
    buscarRiego();
  }, 4000);
}

// ==========================
// Socket.io + Watchdog
// ==========================
function inicializarSocketYWatchdog() {
  const socketio_server_url = `http://${
    window.location.href.split("/")[2]
  }:8080`;
  console.log(
    `WENASHORO: socketio_server_url resolves to [${socketio_server_url}]`
  );
  const socket = io.connect(socketio_server_url, {
    transports: ["websocket"],
  });

  socket.on("connect", () => {
    console.log("connected!");
    const connection_status = socket.connected;
    const user_id = socket.id;
    console.log("socket.io connected, user_id: " + user_id);
  });

  setInterval(function () {
    const present = new Date(Date.now());
    if (typeof window.last_message_timestamp === "undefined") {
      window.last_message_timestamp = new Date(Date.now());
    }
    const last = window.last_message_timestamp;
    const count = (present - last) / 1000;
    if (count > 20) {
      console.log("Reiniciando App luego de " + count);
      if (
        window.AndroidFunction &&
        typeof AndroidFunction.killToast === "function"
      ) {
        AndroidFunction.killToast("Reiniciando App luego de " + count);
      }
    } else {
      console.log("Tiempo incomunicado: " + count);
      // AndroidFunction.showToast("Tiempo incomunicado: " + count);
    }
  }, 1000);
}

// ==========================
// Polling /data para centrar mapa y marcador actual
// ==========================
function inicializarTimerDataCenter() {
  setInterval(function () {
    $.getJSON("/data", function (data) {
      console.log(data);
      eye3.vehicle = data.vehicle;
      console.log(JSON.stringify(data));

      const new_gps_timestamp = new Date();
      if (
        new_gps_timestamp.getTime() - eye3.last_gps_timestamp >
        eye3.map_center_update_interval_ms
      ) {
        eye3.last_gps_timestamp = new_gps_timestamp.getTime();

        console.log("moving map to " + data.lat + "," + data.lon);
        eye3.map.setView(
          [parseFloat(data.lat), parseFloat(data.lon)],
          eye3.map.getZoom(),
          { animation: true }
        );
      } else {
        console.log("Too soon for map re-centering");
      }

      if (typeof eye3.markers[data.vehicle] === "undefined") {
        console.log(
          "No existe marcador para " +
            data.vehicle +
            ". Creando ahora en [" +
            data.lat +
            ", " +
            data.lon +
            "]"
        );
        eye3.markers[data.vehicle] = L.marker([data.lat, data.lon]).addTo(
          eye3.map
        );
        eye3.markers[data.vehicle].setIcon(
          eye3.WaterTruckIcon(data.name, "rgba(75,71,147,0.8)")
        );
      } else {
        eye3.markers[data.vehicle].setIcon(
          eye3.WaterTruckIcon(data.name, "rgba(75,71,147,0.8)")
        );
        eye3.markers[data.vehicle].setLatLng([data.lat, data.lon]);
      }
    }).fail(function () {
      console.error(
        "No se pudo obtener data.json. Llamando a updateMarkers..."
      );
      // En tu código original solo loguea acá, se mantiene así.
    });
  }, 3000);
}
