from flask import Flask, render_template, jsonify, request, redirect, url_for
from filelock import FileLock, Timeout
import json
import subprocess
import os
import requests
from datetime import datetime
import time 
import threading
import sys 
from copy import deepcopy
from werkzeug.datastructures import FileStorage

ruta_keys = os.path.abspath('srv/datalogger_ecom/web-server/autoupdate-data-core')
sys.path.append(ruta_keys)

from keys import API_TOKEN

app = Flask(__name__)

BASE_PATH = "/srv/datalogger_ecom/web-server/"
CABINAS_PATH = f"{BASE_PATH}CABINAS_DATA.json"
EVOS_PATH = f"{BASE_PATH}EVOS_DATA.json"
FAENAS_PATH = f"{BASE_PATH}FAENAS_DATA.json"
USERS_PATH = f"{BASE_PATH}USERS_DATA.json"
DATALOGGERS_PATH = f"{BASE_PATH}DATALOGGER_DATA.json"
PENDING_PATH = f"{BASE_PATH}pending_submissions.json"
PENDING_IMAGES = f"{BASE_PATH}pending_images"

FAENAS = {
    "Antucoya": 1,
    "TallerStgo": 2,
    "Centinela":4,
    "Candelaria":5,
    "Ministro Hales":6,
    "Pelambres":7,
    "Rodomiro Tomic":8,
    "TallerCalama":9
}

FAENAS_UBICACION ={
    "STGO": 16,
    "desarrollo": 102,
    "Test-stgo": 164,
    "TestingStgo": 211,
    "Producción": 212,
    "Taller Calama": 167
}

SENSOR_ESTADO = {
    "Desarrollo": 1,
    "Disponible": 2,
    "Reparación": 3,
    "Instalado": 4,
    "Baja": 6
}

DATALOGGER_ESTADO = {
    "De baja": 1,
    "Disponible": 2,
    "Instalado": 3,
    "Taller Calama": 4,
    "Taller Santiago": 5
}

last_evo_data = {}

## TODO: - AGREGAR PERSONA QUE HIZO EL CAMBIO/INSTALACION
##       - AGREGAR EL ENVIO DE TAREAS
##       - AGREGAR SUBIR FOTO

def get_product_id():
        data = subprocess.check_output(["machineid"]).decode("utf-8").replace("\n", "")
        return int(data)

def get_product_name():
    data = subprocess.check_output(["machinename"]).decode("utf-8").replace("\n", "")
    return data

def get_location_assigned():
    caex = "STGO"
    try:
        f = open('/srv/live/topic.json')
        config:dict = json.load(f)
        caex = config["topic"].split("/")[2]
    except:
        print(f"[ERROR] Fallo al obtener la faena ubicación asignada")
    return caex

def get_faena_assigned():
    faena = "TallerStgo"
    try:
        f = open('/srv/live/topic.json')
        config:dict = json.load(f)
        faena = config["topic"].split("/")[0]
    except:
        print(f"[ERROR] Fallo al obtener la faena asignada")
    return faena

def get_evos_assigned(location, faena):
    try:
        with open("/srv/datalogger_ecom/web-server/LOC_DATA.json", "r", encoding="utf-8") as f:
            config = json.load(f)

        for data in config.values():
            if data.get("FaenaUbicacion") == location and data.get("Faena") == faena:
                return [
                    s for s in data.get("sensores", [])
                    if s.lower().startswith("evo-")
                ]

    except Exception as e:
        print(f"[ERROR] Fallo al obtener los evos asignados: {e}")

    return []

def get_cabinas_assigned(location, faena):
    try:
        with open("/srv/datalogger_ecom/web-server/LOC_DATA.json", "r", encoding="utf-8") as f:
            config = json.load(f)

        for data in config.values():
            if data.get("FaenaUbicacion") == location and data.get("Faena") == faena:
                return [
                    s for s in data.get("sensores", [])
                    if s.lower().startswith("cab-")
                ]

    except Exception as e:
        print(f"[ERROR] Fallo al obtener las cabinas asignadas: {e}")

    return []

def get_ecom_assigned(location, faena):
    try:
        with open("/srv/datalogger_ecom/web-server/LOC_DATA.json", "r", encoding="utf-8") as f:
            config = json.load(f)

        for data in config.values():
            if data.get("FaenaUbicacion") == location and data.get("Faena") == faena:
                return [
                    s for s in data.get("dataloggers", [])
                    if s.lower().startswith("ecom-")
                ]

    except Exception as e:
        print(f"[ERROR] Fallo al obtener los ecom asignados: {e}")

    return []

def get_evos_listened():
    global last_evo_data
    now = int(time.time() * 1000)

    try:
        output = subprocess.check_output(
            f"tail -n 100 /log.txt",
            shell=True
        ).decode("utf-8")

        for line in output.splitlines():
            if "GOT: rx" not in line:
                continue

            contenido = line[43:]
            partes = contenido.split("/")

            if len(partes) != 3:
                continue

            evo, tipo, valor = partes
            evo = evo[1:].strip()
            valor = valor[:-1]

            if tipo != "pm100":
                continue

            if len(evo) > 13 or evo.startswith("["):
                continue

            try:
                valor = int(valor)
            except ValueError:
                try:
                    valor = float(valor)
                except ValueError:
                    continue

            last_evo_data[evo] = {
                "tipo": tipo,
                "valor": valor,
                "timestamp": now
            }

    except Exception as e:
        print(f"[ERROR] Fallo al obtener EVOS: {e}")

    last_evo_data = {
        evo: data
        for evo, data in last_evo_data.items()
        if now - data["timestamp"] <= 60000
    }

    return [
        [evo, data["tipo"], data["valor"], data["timestamp"]]
        for evo, data in last_evo_data.items()
    ]

def get_coords():
    lat, lon = 0.0, 0.0
    try:
        f = open('/gps_live.json')
        config:dict = json.load(f)
        lat = config.get("latitude", 0.0)
        lon = config.get("longitude", 0.0)
    except:
        print(f"[ERROR] Fallo al obtener las coordenadas GPS")
    return lat, lon

def get_dataloggers():
    dataloggers = {}
    try:
        dataloggers = cargar_json(DATALOGGERS_PATH)
        return dataloggers
    except Exception as ex:
        print(f"Error cargando json")

    return dataloggers
        

def cargar_json(path, timeout=5):
    lock = FileLock(path + ".lock")
    try:
        with lock.acquire(timeout=timeout):
            if os.path.exists(path):
                with open(path, "r") as f:
                    return json.load(f)
            return []
    except Timeout:
        print(f"[ERROR] Timeout al leer {path}")
    except Exception as e:
        print(f"[ERROR] Fallo al leer {path}: {e}")
        
def process_submit(form, imagenes):
    faena = form.get("faena")
    ubicacion = form.get("ubicacion")
    evo_reemplazo = form.get("evos")
    cabina_reemplazo = form.get("cabina")
    notas = form.get("notas", "")
    ayudante_1 = form.get("ayudante1", "")
    ayudante_2 = form.get("ayudante2", "")
    tipo_trabajo = 3 #Reemplazo
    responsable = form.get("responsable", "")
    lat, lon = get_coords()
    date = str(datetime.now())
    evo = get_evos_assigned(ubicacion, faena)[0] if get_evos_assigned(ubicacion, faena) else ""
    cab = get_cabinas_assigned(ubicacion, faena)[0] if get_cabinas_assigned(ubicacion, faena) else ""
    ecom = get_ecom_assigned(ubicacion, faena)[0] if get_ecom_assigned(ubicacion, faena) else ""
    ecom_reemplazo = get_product_name()

    files_names = [img.filename for img in imagenes]
    files = [("imagenes", (img.filename, img.read(), img.content_type)) for img in imagenes]
    
    responsable_id = int(responsable)
    users = cargar_json(USERS_PATH)
    responsable_nombre = next(user["nombre_completo"]for user in users.values()if user.get("id") == responsable_id)
    
    data = {
        "status": "success",
        "faena": faena,
        "ubicacion": ubicacion,
        "reemplazo_evo": evo_reemplazo,
        "reemplazo_cab": cabina_reemplazo,
        "notas": notas,
        "imagenes": files_names,
        "ayudante_1": ayudante_1,
        "ayudante_2": ayudante_2,
        "tipo_trabajo": tipo_trabajo,
        "responsable": responsable_nombre,
        "lat": lat,
        "lon": lon,
        "fecha_inicio": date,
        "fecha_fin": date,
        "evo": evo,
        "cab": cab,
        "ecom": ecom,
        "reemplazo_ecom": ecom_reemplazo
    }
    
    api_url = "https://core.mine-360.com/api/trabajos-faenas"
    upload_url = "https://core.mine-360.com/api/upload"
    headers = {"Authorization": f"Bearer {API_TOKEN}"}
    
    ### Trabajos Faena
    
    uploaded_image_ids = []

    if imagenes:
        for img in imagenes:
            if not img or img.filename == "":
                continue

            img.stream.seek(0)

            files = [
                ("files", (img.filename, img.stream, img.content_type))
            ]

            upload_response = requests.post(
                upload_url,
                headers={"Authorization": f"Bearer {API_TOKEN}"},
                files=files,
                timeout=20
            )

            if upload_response.status_code != 200:
                print(upload_response.text)
                upload_response.raise_for_status()

            uploaded_files = upload_response.json()
            uploaded_image_ids.extend([f["id"] for f in uploaded_files])

    faena_id = FAENAS.get(faena, None)
    faena_ubicacion = cargar_json(FAENAS_PATH)
    ubicacion_id = next(item['id'] for item in faena_ubicacion[faena] if item.get('faenaubicacion') == ubicacion)
    dataloggers_dict = get_dataloggers()
    ecom_reemplazo = dataloggers_dict.get(ecom_reemplazo, None)
    ecom = dataloggers_dict.get(ecom, None)
    ayudante_1 = ayudante_1 if ayudante_1 != "" else None
    ayudante_2 = ayudante_2 if ayudante_2 != "" else None
    evos_dict = cargar_json(EVOS_PATH)
    cabinas_dict = cargar_json(CABINAS_PATH)
    evo_reemplazo_id = int(evos_dict[evo_reemplazo]["id"]) if evo_reemplazo != "" else None
    evo_id = int(evos_dict[evo]["id"]) if evo != "" else None
    cabina_reemplazo_id = int(cabinas_dict[cabina_reemplazo]["id"]) if cabina_reemplazo != "" else None
    cab_id = int(cabinas_dict[cab]["id"]) if cab != "" else None

    base_payload = {
        "data": {
            "Faena": faena_id,
            "Ubicacion": ubicacion_id, 
            "Comentario": notas, 
            "Ayudante1": ayudante_1, 
            "Ayudante2": ayudante_2, 
            "tipo_trabajo": tipo_trabajo, 
            "Responsable": responsable, 
            "users_permissions_user": responsable, 
            "lat": lat, 
            "lon": lon, 
            "FechaInicio": date, 
            "FechaFin": date,
            "Imagen": uploaded_image_ids, 
            "publishedAt": date
        }}

    if evo_reemplazo_id and cabina_reemplazo_id:
        payload_1 = deepcopy(base_payload)
        payload_1["data"].update({
            "ReemplazoSensor": evo_reemplazo_id,
            "sensor": evo_id,
            "Equipo": ecom,
            "Reemplazo": ecom_reemplazo
        })

        payload_2 = deepcopy(base_payload)
        payload_2["data"].update({
            "ReemplazoSensor": cabina_reemplazo_id,
            "sensor": cab_id
        })
        response1 = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload_1, timeout=20)
        response2 = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload_2, timeout=20)
        response1.raise_for_status()
        response2.raise_for_status()
    
    elif evo_reemplazo_id and not cabina_reemplazo_id:
        payload = deepcopy(base_payload)
        payload["data"].update({
            "ReemplazoSensor": evo_reemplazo_id,
            "sensor": evo_id,
            "Equipo": ecom,
            "Reemplazo": ecom_reemplazo
        })
        response = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload, timeout=20)
        response.raise_for_status()
    
    elif not evo_reemplazo_id and cabina_reemplazo_id:
        payload = deepcopy(base_payload)
        payload["data"].update({
            "ReemplazoSensor": cabina_reemplazo_id,
            "sensor": cab_id,
            "Equipo": ecom,
            "Reemplazo": ecom_reemplazo
        })
        response = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload, timeout=20)
        response.raise_for_status()
    
    elif not evo_reemplazo_id and not cabina_reemplazo_id:
        payload = deepcopy(base_payload)
        payload["data"].update({
            "Equipo": ecom,
            "Reemplazo": ecom_reemplazo
        })
        response = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload, timeout=20)
        response.raise_for_status()
        
    ### Faena Ubicación 
    
    f_ubicacion_url = f"https://core.mine-360.com/api/faena-ubicacions/{ubicacion_id}"
    
    # pensar que se usará en ecom y mmr
    dataloggers = []
    
    if ecom_reemplazo: 
        dataloggers.append(ecom_reemplazo)
    
    elif ecom: 
        dataloggers.append(ecom)
        
    sensors = []

    if evo_reemplazo_id:
        sensors.append(evo_reemplazo_id)
    elif evo_id:
        sensors.append(evo_id)

    if cabina_reemplazo_id:
        sensors.append(cabina_reemplazo_id)
    elif cab_id:
        sensors.append(cab_id)

    
    f_ubicacion_payload = {
        "data": {
            "dataloggers": dataloggers,
            "sensores": sensors
        }
    }

    response = requests.put(f_ubicacion_url, headers={**headers, "Content-Type": "application/json"}, json=f_ubicacion_payload, timeout=20)
    response.raise_for_status()
    
    ### ECOM
    if ecom != ecom_reemplazo and ecom:
        datalogger_url = f"https://core.mine-360.com/api/dataloggers/{ecom}"
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{ecom_reemplazo}"
        
        datalogger_payload = {
            "data": {
                "faena": FAENAS["TallerStgo"],
                "faena_ubicacion": [],
                "datalogger_estado": DATALOGGER_ESTADO["Disponible"]
            }
        }
        #print(f"ECOM SALIENTE: {datalogger_payload}")
        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"ECOM 1 ENTRANTE: {datalogger_reemplazo_payload}")
        response1 = requests.put(datalogger_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_payload, timeout=20)
        response1.raise_for_status()
        response2 = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        response2.raise_for_status()
    elif ecom != ecom_reemplazo:
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{ecom_reemplazo}"
        
        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"ECOM 2 ENTRANTE: {datalogger_reemplazo_payload}")
        response = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        print("hola")
        response.raise_for_status()
    
    ### Sensores
    sensor_payload = {
        "data": {
            "faena": FAENAS["TallerStgo"], 
            "faena_ubicacion": [], 
            "sensor_estado": SENSOR_ESTADO["Disponible"]
        }
    }
    
    sensor_reemplazo_payload = {
        "data": {
            "faena": faena_id, 
            "faena_ubicacion": ubicacion_id, 
            "sensor_estado": SENSOR_ESTADO["Instalado"]
        }
    }
    
    if evo_id != evo_reemplazo_id and evo_id and evo_reemplazo_id: 
        evo_url = f"https://core.mine-360.com/api/sensores/{evo_id}"
        evo_reemplazo_url = f"https://core.mine-360.com/api/sensores/{evo_reemplazo_id}"
        response1 = requests.put(evo_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_payload, timeout=100)
        response1.raise_for_status()
        response2 = requests.put(evo_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_reemplazo_payload, timeout=100)
        response2.raise_for_status()

            
    elif evo_id != evo_reemplazo_id and evo_reemplazo_id:
        evo_reemplazo_url = f"https://core.mine-360.com/api/sensores/{evo_reemplazo_id}"
        
        response = requests.put(evo_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_reemplazo_payload, timeout=100)
        response.raise_for_status()
        
    if cab_id != cabina_reemplazo_id and cab_id and cabina_reemplazo_id: 
        cab_url = f"https://core.mine-360.com/api/sensores/{cab_id}"
        cab_reemplazo_url = f"https://core.mine-360.com/api/sensores/{cabina_reemplazo_id}"
        
        response1 = requests.put(cab_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_payload, timeout=100)
        response1.raise_for_status()
        response2 = requests.put(cab_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_reemplazo_payload, timeout=100)
        response2.raise_for_status()
    elif cab_id != cabina_reemplazo_id and cabina_reemplazo_id: 
        cab_reemplazo_url = f"https://core.mine-360.com/api/sensores/{cabina_reemplazo_id}"
        
        response = requests.put(cab_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=sensor_reemplazo_payload, timeout=100)
        response.raise_for_status()
        
    print("Preparando actualización de datos")
    subprocess.check_output(["systemctl", "restart", "mining-autoupdate-data-core.service"]) #Comentado para pruebas locales
    subprocess.check_output(["topic"])
    print("Datos actualizados")
    
    return jsonify(data)

def guardar_pendientes(data, files):
    lock = FileLock(PENDING_PATH + ".lock")
    try:
        with lock.acquire(timeout=5):
            path_imagenes = []
            timestamp = datetime.now().strftime("%Y%m%d%H%M%S%f")
            for i, img in enumerate(files):
                extension = os.path.splitext(img.filename)[1]
                filename = f"{timestamp}_{i}{extension}"
                filepath = os.path.join(PENDING_IMAGES, filename)
                img.save(filepath)
                path_imagenes.append(filepath)
                
            entry = {
                "form": dict(data),
                "images_paths": path_imagenes
            }

            data_pendiente = []

            if os.path.exists(PENDING_PATH):
                with open(PENDING_PATH, "r") as f:
                    data_pendiente = json.load(f)

            data_pendiente.append(entry)

            with open(PENDING_PATH, "w") as f:
                json.dump(data_pendiente, f, indent=2)

    except Exception as e:
        print(f"[ERROR] No se pudo guardar pendiente: {e}")

def reenviar_pendientes():
    while True:
        if not has_internet_connection():
            time.sleep(60)
            continue
        
        pendientes = cargar_json(PENDING_PATH)
        if not pendientes:
            time.sleep(60)
            continue

        for entry in pendientes.copy(): 
            try:
                imagenes = []
                for path in entry["images_paths"]:
                    if os.path.exists(path):
                        imagenes.append(
                            FileStorage(
                                stream=open(path, "rb"),
                                filename=os.path.basename(path),
                                content_type="image/jpeg"
                            )
                        )

                process_submit(entry["form"], imagenes)

                for path in entry["images_paths"]:
                    if os.path.exists(path):
                        os.remove(path)

                pendientes.remove(entry)

            except Exception as e:
                print(f"[ERROR] No se pudo reenviar: {e}")

        lock = FileLock(PENDING_PATH + ".lock")
        try: 
            with lock.acquire(timeout=5):
                with open(PENDING_PATH, "w") as f:
                    json.dump(pendientes, f, indent=2)
        except Exception as e:
            print(f"[ERROR] No se pudo guardar la lista de pendientes: {e}")
        time.sleep(60)

# threading.Thread(target=reenviar_pendientes, daemon=True).start()

def has_internet_connection():
    try:
        requests.get("https://www.google.com", timeout=3)
        return True
    except requests.RequestException:
        return False

@app.route('/')
def home():
    return render_template('index.html')

@app.route('/get_machine_data', methods=['GET']) ## Estado de cargar agua o surf
def get_machine():
    subprocess.check_output(["topic"])
    data = {
        "machine_name": "",
        "machine_id": "",
        "faena": "",
        "ubicacion": "",
        "evos": []
    }
    try:
        machine_name = get_product_name()
        machine_id = get_product_id()
        faena = get_faena_assigned()
        location = get_location_assigned()
        evos = get_evos_assigned(location=location, faena=faena)

        data["machine_name"] = machine_name
        data["machine_id"] = machine_id
        data["faena"] = faena
        data["ubicacion"] = location
        data["evos"] = evos

        return jsonify(data)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(data)

@app.route('/get_cabinas', methods=['GET']) ## Estado de cargar agua o surf
def get_cabinas():
    cabinas = {}
    try:
        cabinas = cargar_json(CABINAS_PATH)
        return jsonify(cabinas)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(cabinas)

@app.route('/get_evos', methods=['GET']) ## Estado de cargar agua o surf
def get_evos():
    evos = {}
    try:
        evos = cargar_json(EVOS_PATH)
        return jsonify(evos)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(evos)

@app.route('/get_faenas', methods=['GET']) ## Estado de cargar agua o surf
def get_faenas():
    faenas = {}
    try:
        faenas = cargar_json(FAENAS_PATH)
        #print(faenas)
        return jsonify(faenas)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(faenas)

@app.route('/get_users', methods=['GET']) ## Estado de cargar agua o surf
def get_users():
    users = {}
    try:
        users = cargar_json(USERS_PATH)
        #print(users)
        return jsonify(users)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(users)

@app.route('/get_evos_listened', methods=['GET']) ## Estado de cargar agua o surf
def get_evos_listened_route():
    evos = []
    try:
        evos = get_evos_listened()
        return jsonify(evos)
    except Exception as ex:
        print(f"Error obteniendo evos escuchados: {ex}")

    return jsonify(evos)

@app.route('/actualizar', methods = ['POST'])
def actualizar():
    try: 
        print("Actualización solicitada")
        subprocess.check_output(["systemctl", "restart", "mining-autoupdate-data-core.service"]) #Comentar para pruebas locales
        print("Actualización exitosa")
        return jsonify({"status": "success", "message": "Datos actualizados"})
    except Exception as e: 
        print(f"Error al actualizar {e}")
        return jsonify({"status": "error", "message": str(e)}), 500

@app.route('/submit', methods=['POST'])
def submit():
    #if not has_internet_connection():
    #    data = request.form
    #    imagenes = request.files.getlist("imagenes")
    #    guardar_pendientes(data, imagenes)
    #    return jsonify({
    #        "status": "error",
    #        "message": "No hay conexión a internet. Datos guardados localmente. Se enviarán cuando haya conexión."
    #    }), 503
    try:
        form = request.form
        imagenes = request.files.getlist("imagenes")
        return process_submit(form, imagenes)

    except Exception as ex:
        print(f"[ERROR] Fallo en /submit: {ex}")
        return jsonify({
            "status": "error",
            "message": str(ex)
        }), 500

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)