from flask import Flask, render_template, jsonify, request, redirect, url_for
from filelock import FileLock, Timeout
import json
import subprocess
import os
import requests
from datetime import datetime
import time 
import threading
import sys 
from copy import deepcopy
from werkzeug.datastructures import FileStorage

ruta_keys = os.path.abspath('srv/datalogger_mmr/web-server/autoupdate-data-core')
sys.path.append(ruta_keys)

from keys import API_TOKEN

app = Flask(__name__)

BASE_PATH = "/srv/datalogger_mmr/web-server/"
FAENAS_PATH = f"{BASE_PATH}FAENAS_DATA.json"
USERS_PATH = f"{BASE_PATH}USERS_DATA.json"
DATALOGGERS_PATH = f"{BASE_PATH}DATALOGGER_DATA.json"
PENDING_PATH = f"{BASE_PATH}pending_submissions.json"
PENDING_IMAGES = f"{BASE_PATH}pending_images"

FAENAS = {
    "Antucoya": 1,
    "TallerStgo": 2,
    "Centinela":4,
    "Candelaria":5,
    "Ministro Hales":6,
    "Pelambres":7,
    "Rodomiro Tomic":8,
    "TallerCalama":9
}

FAENAS_UBICACION ={
    "STGO": 16,
    "desarrollo": 102,
    "Test-stgo": 164,
    "TestingStgo": 211,
    "Producción": 212,
    "Taller Calama": 167
}

SENSOR_ESTADO = {
    "Desarrollo": 1,
    "Disponible": 2,
    "Reparación": 3,
    "Instalado": 4,
    "Baja": 6
}

DATALOGGER_ESTADO = {
    "De baja": 1,
    "Disponible": 2,
    "Instalado": 3,
    "Taller Calama": 4,
    "Taller Santiago": 5
}

def get_product_id():
    try:
        data = subprocess.check_output(["machineid"]).decode("utf-8").replace("\n", "")
        return int(data)
    except:
        print(f"[ERROR] Fallo al obtener el ID del producto")
    return 0

def get_product_name():
    try:
        data = subprocess.check_output(["machinename"]).decode("utf-8").replace("\n", "")
        return data
    except:
        print(f"[ERROR] Fallo al obtener el nombre del producto")
    return "N/A"

def get_location_assigned():
    reg = "STGO"
    try:
        f = open('/srv/live/topic.json')
        config:dict = json.load(f)
        reg = config["topic"].split("/")[2]
    except:
        print(f"[ERROR] Fallo al obtener la faena ubicación asignada")
    return reg

def get_faena_assigned():
    faena = "TallerStgo"
    try:
        f = open('/srv/live/topic.json')
        config:dict = json.load(f)
        faena = config["topic"].split("/")[0]
    except:
        print(f"[ERROR] Fallo al obtener la faena asignada")
    return faena

def get_mmr_asigned(location):
    try:
        with open("/srv/web-server/LOC_DATA.json", "r", encoding="utf-8") as f:
            config = json.load(f)

        for data in config.values():
            if data.get("FaenaUbicacion") == location:
                return [
                    s for s in data.get("dataloggers", [])
                    if s.lower().startswith("mmr-")
                ]

    except Exception as e:
        print(f"[ERROR] Fallo al obtener los mmr asignados: {e}")

    return []

def get_tablet_asigned(location):
    try:
        with open("/srv/web-server/LOC_DATA.json", "r", encoding="utf-8") as f:
            config = json.load(f)

        for data in config.values():
            if data.get("FaenaUbicacion") == location:
                return [
                    s for s in data.get("dataloggers", [])
                    if s.lower().startswith("tablet-")
                ]

    except Exception as e:
        print(f"[ERROR] Fallo al obtener las tablets asignadas: {e}")

    return []

def get_coords():
    lat, lon = 0.0, 0.0
    try:
        f = open('/gps_live.json')
        config:dict = json.load(f)
        lat = config.get("latitude", 0.0)
        lon = config.get("longitude", 0.0)
    except:
        print(f"[ERROR] Fallo al obtener las coordenadas GPS")
    return lat, lon

def get_dataloggers():
    dataloggers = {}
    try:
        dataloggers = cargar_json(DATALOGGERS_PATH)
        return dataloggers
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return dataloggers 

def get_log_data():
    data = {
        "date": None,
        "time": None,
        "level": None, 
        "ltr": None,
        "ltc": None,
        "rtr": None,
        "rtc": None,
    }
    try: 
        output = subprocess.check_output(f"tail -n 100 /log.txt", shell=True).decode("utf-8")
        for line in output.splitlines():
            if not line.startswith("[SERIAL]"):
                continue
            data["date"] = line.split()[1]
            data["time"] = line.split()[2]
            if "M_LTR:" in line:
                data["ltr"] = int(line.split("M_LTR:")[1])
            elif "M_LTC:" in line:
                data["ltc"] = int(line.split("M_LTC:")[1])
            if "M_RTR:" in line:
                data["rtr"] = int(line.split("M_RTR:")[1])
            elif "M_RTC:" in line:
                data["rtc"] = int(line.split("M_RTC:")[1])
            elif "NIVEL_ESTANQUE:" in line:
                data["level"] = int(line.split("NIVEL_ESTANQUE:")[1].strip())
            
    except Exception as e: 
        print(f"[ERROR] Fallo al obtener data: {e}")
        
    return data

def cargar_json(path, timeout=5):
    lock = FileLock(path + ".lock")
    try:
        with lock.acquire(timeout=timeout):
            if os.path.exists(path):
                with open(path, "r") as f:
                    return json.load(f)
            return []
    except Timeout:
        print(f"[ERROR] Timeout al leer {path}")
    except Exception as e:
        print(f"[ERROR] Fallo al leer {path}: {e}")
        
def process_submit(form, imagenes):
    faena = form.get("faena")
    ubicacion = form.get("ubicacion")
    tablet_reemplazo = form.get("tablet")
    notas = form.get("notas", "")
    ayudante_1 = form.get("ayudante1", "")
    ayudante_2 = form.get("ayudante2", "")
    tipo_trabajo = 3 #Reemplazo
    responsable = form.get("responsable", "")
    lat, lon = get_coords()
    date = str(datetime.now())
    mmrs = get_mmr_asigned(ubicacion) or []
    tablets = get_tablet_asigned(ubicacion) or []
    mmr = mmrs[0] if mmrs else ""
    tablet = tablets[0] if tablets else ""
    mmr_reemplazo = get_product_name()
    
    files_names = [img.filename for img in imagenes]
    files = [("imagenes", (img.filename, img.read(), img.content_type)) for img in imagenes]

    responsable_id = int(responsable)
    users = cargar_json(USERS_PATH)
    responsable_nombre = next(user["nombre_completo"]for user in users.values()if user.get("id") == responsable_id)
    
    data = {
        "status": "success",
        "faena": faena,
        "ubicacion": ubicacion,
        "notas": notas,
        "imagenes": files_names,
        "ayudante_1": ayudante_1,
        "ayudante_2": ayudante_2,
        "tipo_trabajo": tipo_trabajo,
        "responsable": responsable_nombre,
        "lat": lat,
        "lon": lon,
        "fecha_inicio": date,
        "fecha_fin": date,
        "mmr": mmr,
        "reemplazo_mmr": mmr_reemplazo,
        "tablet": tablet,
        "reemplazo_tablet": tablet_reemplazo
    }
    
    api_url = "https://core.mine-360.com/api/trabajos-faenas"
    upload_url = "https://core.mine-360.com/api/upload"
    headers = {"Authorization": f"Bearer {API_TOKEN}"}
    
    ### Trabajos Faena
    
    uploaded_image_ids = []
    
    if imagenes:
        for img in imagenes:
            if not img or img.filename == "":
                continue

            img.stream.seek(0)

            files = [
                ("files", (img.filename, img.stream, img.content_type))
            ]

            upload_response = requests.post(
                upload_url,
                headers=headers,
                files=files,
                timeout=20
            )

            if upload_response.status_code != 200:
                print(upload_response.text)
                upload_response.raise_for_status()

            uploaded_files = upload_response.json()
            uploaded_image_ids.extend([f["id"] for f in uploaded_files])

    faena_id = FAENAS.get(faena, None)
    faena_ubicacion = cargar_json(FAENAS_PATH)
    ubicacion_id = next(item['id'] for item in faena_ubicacion[faena] if item.get('faenaubicacion') == ubicacion)
    dataloggers_dict = get_dataloggers()
    tablet_reemplazo = dataloggers_dict.get(tablet_reemplazo, None)
    tablet = dataloggers_dict.get(tablet, None)
    mmr_reemplazo = dataloggers_dict.get(mmr_reemplazo, None)
    mmr = dataloggers_dict.get(mmr, None)
    ayudante_1 = ayudante_1 if ayudante_1 != "" else None
    ayudante_2 = ayudante_2 if ayudante_2 != "" else None
    
    base_payload = {
        "data": {
            "Faena": faena_id,
            "Ubicacion": ubicacion_id, 
            "Comentario": notas, 
            "Ayudante1": ayudante_1, 
            "Ayudante2": ayudante_2, 
            "tipo_trabajo": tipo_trabajo, 
            "Responsable": responsable, 
            "users_permissions_user": responsable, 
            "lat": lat, 
            "lon": lon, 
            "FechaInicio": date, 
            "FechaFin": date,
            "Imagen": uploaded_image_ids, 
            "publishedAt": date
        }}
    
    if mmr != mmr_reemplazo:
        payload = deepcopy(base_payload)
        payload["data"].update({
            "Equipo": mmr,
            "Reemplazo": mmr_reemplazo
        })
        
        response = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload, timeout=20)
        response.raise_for_status()
    
    if tablet != tablet_reemplazo:
        payload = deepcopy(base_payload)
        payload["data"].update({
            "Equipo": tablet,
            "Reemplazo": tablet_reemplazo
        })

        response = requests.post(api_url, headers={**headers, "Content-Type": "application/json"}, json=payload, timeout=20)
        response.raise_for_status()
    
    ### Faena Ubicación
    
    f_ubicacion_url = f"https://core.mine-360.com/api/faena-ubicacions/{ubicacion_id}"
    
    dataloggers = []
    
    if mmr_reemplazo: 
        dataloggers.append(mmr_reemplazo)
    
    elif mmr: 
        dataloggers.append(mmr)
    
    if tablet_reemplazo:
        dataloggers.append(tablet_reemplazo)
    
    elif tablet:
        dataloggers.append(tablet)
    
    f_ubicacion_payload = {
        "data": {
            "dataloggers": dataloggers,
        }
    }
    
    response = requests.put(f_ubicacion_url, headers={**headers, "Content-Type": "application/json"}, json=f_ubicacion_payload, timeout=20)
    response.raise_for_status()
    
    ## MMR
    
    mmr_id = mmr["id"]
    mmr_reemplazo_id = mmr_reemplazo["id"]
    if mmr != mmr_reemplazo and mmr:
        datalogger_url = f"https://core.mine-360.com/api/dataloggers/{mmr_id}"
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{mmr_reemplazo_id}"

        datalogger_payload = {
            "data": {
                "faena": FAENAS["TallerStgo"],
                "faena_ubicacion": [],
                "datalogger_estado": DATALOGGER_ESTADO["Disponible"]
            }
        }
        #print(f"MMR SALIENTE: {datalogger_payload}")
        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"MMR 1 ENTRANTE: {datalogger_reemplazo_payload}")
        response1 = requests.put(datalogger_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_payload, timeout=20)
        response1.raise_for_status()
        response2 = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        response2.raise_for_status()
    elif mmr != mmr_reemplazo:
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{mmr_reemplazo_id}"

        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"MMR 2 ENTRANTE: {datalogger_reemplazo_payload}")
        response = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        response.raise_for_status()
        
    ### Tablet
    
    tablet_id = tablet["id"]
    tablet_reemplazo_id = tablet_reemplazo["id"]
    if tablet != tablet_reemplazo and tablet and tablet_reemplazo:
        datalogger_url = f"https://core.mine-360.com/api/dataloggers/{tablet_id}"
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{tablet_reemplazo_id}"

        datalogger_payload = {
            "data": {
                "faena": FAENAS["TallerStgo"],
                "faena_ubicacion": [],
                "datalogger_estado": DATALOGGER_ESTADO["Disponible"]
            }
        }
        #print(f"Tablet SALIENTE: {datalogger_payload}")
        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"Tablet 1 ENTRANTE: {datalogger_reemplazo_payload}")
        response1 = requests.put(datalogger_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_payload, timeout=20)
        response1.raise_for_status()
        response2 = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        response2.raise_for_status()
    elif tablet != tablet_reemplazo and tablet_reemplazo:
        datalogger_reemplazo_url = f"https://core.mine-360.com/api/dataloggers/{tablet_reemplazo_id}"

        datalogger_reemplazo_payload = {
            "data": {
                "faena": faena_id,
                "faena_ubicacion": ubicacion_id,
                "datalogger_estado": DATALOGGER_ESTADO["Instalado"]
            }
        }
        #print(f"Tablet 2 ENTRANTE: {datalogger_reemplazo_payload}")
        response = requests.put(datalogger_reemplazo_url, headers={**headers, "Content-Type": "application/json"}, json=datalogger_reemplazo_payload, timeout=20)
        response.raise_for_status()
    
    print("Preparando actualización de datos")
    subprocess.check_output(["systemctl", "restart", "mining-autoupdate-data-core.service"]) #Comentado para pruebas locales
    subprocess.check_output(["topic"])
    print("Datos actualizados")
    
    return jsonify(data)

def guardar_pendientes(data, files):
    lock = FileLock(PENDING_PATH + ".lock")
    try:
        with lock.acquire(timeout=5):
            path_imagenes = []
            timestamp = datetime.now().strftime("%Y%m%d%H%M%S%f")
            for i, img in enumerate(files):
                extension = os.path.splitext(img.filename)[1]
                filename = f"{timestamp}_{i}{extension}"
                filepath = os.path.join(PENDING_IMAGES, filename)
                img.save(filepath)
                path_imagenes.append(filepath)

            entry = {
                "form": dict(data),
                "images_paths": path_imagenes
            }
            
            data_pendiente = []
            
            if os.path.exists(PENDING_PATH):
                with open(PENDING_PATH, "r") as f:
                    data_pendiente = json.load(f)

            data_pendiente.append(entry)

            with open(PENDING_PATH, "w") as f:
                json.dump(data_pendiente, f, indent=2)

    except Exception as e:
        print(f"[ERROR] No se pudo guardar pendiente: {e}")

def reenviar_pendientes():
    while True:
        if not has_internet_connection():
            time.sleep(60)
            continue
        
        pendientes = cargar_json(PENDING_PATH)
        if not pendientes:
            time.sleep(60)
            continue
        
        for entry in pendientes.copy(): 
            try: 
                imagenes = []
                for path in entry["images_paths"]:
                    if os.path.exists(path):
                        imagenes.append(
                            FileStorage(
                                stream=open(path, "rb"),
                                filename=os.path.basename(path),
                                content_type="image/jpeg"
                            )
                        )
                    
                    process_submit(entry["form"], imagenes)
                    
                    for path in entry["images_paths"]:
                        if os.path.exists(path):
                            os.remove(path)
                    
                    pendientes.remove(entry)
                    
            except Exception as e:
                print(f"[ERROR] No se pudo reenviar: {e}")
            
        lock = FileLock(PENDING_PATH + ".lock")
        try:
            with lock.acquire(timeout=5):
                with open(PENDING_PATH, "w") as f:
                    json.dump(pendientes, f, indent=2)
        except Exception as e:
            print(f"[ERROR] No se pudo guardar la lista de pendientes: {e}")

# threading.Thread(target=reenviar_pendientes, daemon=True).start()

def has_internet_connection():
    try:
        requests.get("https://www.google.com", timeout=3)
        return True
    except requests.RequestException:
        return False
    
@app.route('/')
def home():
    return render_template('index.html')

@app.route('/get_machine_data', methods=['GET']) ## Estado de cargar agua o surf
def get_machine():
    subprocess.check_output(["topic"])
    data = {
        "machine_name": "",
        "machine_id": "",
        "faena": "",
        "ubicacion": "",
    }
    try:
        machine_name = get_product_name()
        machine_id = get_product_id()
        faena = get_faena_assigned()
        location = get_location_assigned()

        data["machine_name"] = machine_name
        data["machine_id"] = machine_id
        data["faena"] = faena
        data["ubicacion"] = location

        return jsonify(data)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(data)

@app.route('/get_faenas', methods=['GET']) ## Estado de cargar agua o surf
def get_faenas():
    faenas = {}
    try:
        faenas = cargar_json(FAENAS_PATH)
        #print(faenas)
        return jsonify(faenas)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(faenas)

@app.route('/get_users', methods=['GET']) ## Estado de cargar agua o surf
def get_users():
    users = {}
    try:
        users = cargar_json(USERS_PATH)
        #print(users)
        return jsonify(users)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(users)

@app.route('/get_dataloggers', methods=['GET']) ## Estado de cargar agua o surf
def get_dataloggers_data():
    dataloggers = {}
    try:
        dataloggers = cargar_json(DATALOGGERS_PATH)
        #print(dataloggers)
        return jsonify(dataloggers)
    except Exception as ex:
        print(f"Error cargando json: {ex}")

    return jsonify(dataloggers)

@app.route('/get_log_data', methods=['GET']) 
def get_log():
    log_data = {}
    try: 
        log_data = get_log_data()
        return jsonify(log_data)
    except Exception as e: 
        print(f"Error al obtener log data: {e}")
    
    return jsonify(log_data)

@app.route('/actualizar', methods = ['POST'])
def actualizar():
    try: 
        print("Actualización solicitada")
        subprocess.check_output(["systemctl", "restart", "mining-autoupdate-data-core.service"]) #Comentar para pruebas locales
        print("Actualización exitosa")
        return jsonify({"status": "success", "message": "Datos actualizados"})
    except Exception as e: 
        print(f"Error al actualizar {e}")
        return jsonify({"status": "error", "message": str(e)}), 500
    
@app.route('/submit', methods=['POST'])
def submit():
    #if not has_internet_connection():
    #    data = request.form
    #    imagenes = request.files.getlist("imagenes")
    #    guardar_pendientes(data, imagenes)
    #    return jsonify({
    #        "status": "error",
    #        "message": "No hay conexión a internet. Datos guardados localmente. Se enviarán cuando haya conexión."
    #    }), 503
    try:
        form = request.form
        imagenes = request.files.getlist("imagenes")
        return process_submit(form, imagenes)

    except Exception as ex:
        print(f"[ERROR] Fallo en /submit: {ex}")
        return jsonify({
            "status": "error",
            "message": str(ex)
        }), 500
        
if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)