function $(selector) {
    return document.querySelector(selector);
}

const app = document.getElementById("app");

/* =========================
   TOAST
========================= */
function showToast(message, success = true, duration = 1500) {
    const toast = document.createElement("div");
    toast.textContent = message;

    Object.assign(toast.style, {
        position: "fixed",
        top: "30px",
        right: "30px",
        padding: "10px 20px",
        backgroundColor: success ? "#28a745" : "#dc3545",
        color: "white",
        fontWeight: "bold",
        borderRadius: "6px",
        zIndex: 9999,
    });

    document.body.appendChild(toast);
    setTimeout(() => toast.remove(), duration);
}

/* =========================
   TITULO MAQUINA
========================= */
const h1 = document.createElement("h1");
h1.textContent = "Cargando máquina...";
app.appendChild(h1);

const h2 = document.createElement("h2");
app.appendChild(h2);

/* =========================
   TARJETAS LOG SERIAL
========================= */
const logCards = document.createElement("div");
logCards.classList.add("log-cards-container");

logCards.innerHTML = `
    <div class="log-header">
        <div class="log-title">📡 Último dato obtenido</div>
        <div class="log-datetime">Cargando...</div>
    </div>

    <div class="log-grid">
        <div class="card" id="card-ltr"></div>
        <div class="card" id="card-ltc"></div>
        <div class="card" id="card-rtr"></div>
        <div class="card" id="card-rtc"></div>
    </div>

    <div class="card card-level" id="card-level"></div>
`;

app.appendChild(logCards);

async function get_machine_data() {
    try {
        const res = await fetch("/get_machine_data");
        const json = await res.json();

        return {
            machine_name: json.machine_name || "N/A",
            machine_id: json.machine_id || "N/A",
            faena: json.faena || "N/A",
            ubicacion: json.ubicacion || "N/A",
            evos: json.evos || []
        };
    } catch (err) {
        console.error(err);
        return {
            machine_name: "N/A",
            machine_id: "N/A",
            faena: "N/A",
            ubicacion: "N/A",
            evos: []
        };
    }
}

function formatDateTimeUTC(dateStr, timeStr) {
    if (!dateStr || !timeStr) return "-";

    // 👇 Z = UTC explícito
    const d = new Date(`${dateStr}T${timeStr}Z`);

    return d.toLocaleString("es-CL", {
        timeZone: "America/Santiago",
        hour: "2-digit",
        minute: "2-digit",
        second: "2-digit",
        day: "2-digit",
        month: "2-digit",
        year: "numeric"
    });
}

function estadoCard(valor) {
    return valor === 1
        ? `<span class="state on">1</span>`
        : `<span class="state off">0</span>`;
}

async function cargarResumenLog() {
    try {
        const res = await fetch("/get_log_data");
        const d = await res.json();

        document.querySelector(".log-datetime").textContent =
            formatDateTimeUTC(d.date, d.time);


        $("#card-ltr").innerHTML = `
            <div class="label"><b>M_LTR</b></div>
            <div class="value">${estadoCard(d.ltr)}</div>
        `;

        $("#card-ltc").innerHTML = `
            <div class="label"><b>M_LTC</b></div>
            <div class="value">${estadoCard(d.ltc)}</div>
        `;

        $("#card-rtr").innerHTML = `
            <div class="label"><b>M_RTR</b></div>
            <div class="value">${estadoCard(d.rtr)}</div>
        `;

        $("#card-rtc").innerHTML = `
            <div class="label"><b>M_RTC</b></div>
            <div class="value">${estadoCard(d.rtc)}</div>
        `;

        $("#card-level").innerHTML = `
            <div class="label"><b>Nivel estanque</b></div>
            <div class="value level">${d.level ?? "-"} %</div>
        `;

    } catch (err) {
        console.error("Error cargando resumen log", err);
        document.querySelector(".log-datetime").textContent =
            "❌ Error obteniendo datos";
    }
}


async function cargarTitulo() {
    const data = await get_machine_data();
    h1.textContent = `${data.machine_name} / ID ${data.machine_id}`;

    if (!data.faena || !data.ubicacion) {
        h2.innerHTML = "❌ Equipo sin asignación";
        h2.className = "sin-asignacion";
    } else {
        h2.innerHTML = `✔️ ${data.ubicacion} — ${data.faena}`;
        h2.className = "asignado";
    }
}

cargarTitulo();

cargarResumenLog();

setInterval(cargarResumenLog, 2000); 

/* =========================
   RESPONSABLE
========================= */
const formResponsable = document.createElement("form");
formResponsable.innerHTML = `
<h2>Seleccionar Responsable</h2>
<select id="responsable" required>
    <option value="">Seleccionar Responsable</option>
</select>
<button class="btn-submit">Continuar</button>
`;
app.appendChild(formResponsable);

let RESPONSABLE_FINAL = null;

async function cargarResponsables() {
    try {
        const res = await fetch("/get_users");
        const data = await res.json();
        //console.log("Usuarios recibidos:", data);

        let usuarios = [];

        if (Array.isArray(data)) {
            usuarios = data;
        } else if (typeof data === "object" && data !== null) {
            usuarios = Object.values(data);
        }

        usuarios = usuarios
            .filter(u => u && u.id && u.nombre_completo)
            .sort((a, b) =>
                a.nombre_completo.localeCompare(b.nombre_completo)
            );

        const sel = $("#responsable");
        sel.innerHTML = `<option value="">Seleccionar Responsable</option>`;

        usuarios.forEach(u => {
            const opt = document.createElement("option");
            opt.value = u.id;
            opt.textContent = u.nombre_completo;
            sel.appendChild(opt);
        });

    } catch (err) {
        console.error("Error cargando responsables:", err);
        showToast("Error cargando responsables", false);
    }
}


cargarResponsables();

formResponsable.onsubmit = e => {
    e.preventDefault();
    RESPONSABLE_FINAL = $("#responsable").value;
    if (!RESPONSABLE_FINAL) return;

    formResponsable.style.display = "none";
    form.style.display = "block";
};

/* =========================
   FORMULARIO
========================= */
const form = document.createElement("form");
form.action = "/submit";
form.method = "POST";
form.style.display = "none";
app.appendChild(form);

function crearSelect(id, name, opciones = [], labelText = "", obligatorio = false) {
    const div = document.createElement("div");

    const select = document.createElement("select");
    select.id = id;
    select.name = name;
    if (obligatorio) select.required = true; 

    const label = document.createElement("label");
    label.htmlFor = id;
    label.textContent = labelText;
    div.appendChild(label);

    opciones.forEach(op => {
        const option = document.createElement("option");
        option.value = op.value;
        option.textContent = op.text;
        select.appendChild(option);
    });

    div.appendChild(select);
    return div;
}

const tituloAsignacion = document.createElement("h2");
tituloAsignacion.textContent = "ASIGNACIÓN";
tituloAsignacion.classList.add("asignacion-titulo"); 
form.appendChild(tituloAsignacion);

form.appendChild(
    crearSelect("ayudante1", "ayudante1", [{value: "", text: "Seleccionar Ayudante 1"}], "AYUDANTE 1")
);

form.appendChild(
    crearSelect("ayudante2", "ayudante2", [{value: "", text: "Seleccionar Ayudante 2"}], "AYUDANTE 2")
);
form.appendChild(
    crearSelect("faena", "faena", [{ value: "", text: "Seleccionar Faena" }], "FAENA", true)
);

form.appendChild(
    crearSelect("ubicacion", "ubicacion", [{ value: "", text: "Seleccionar Ubicación" }], "UBICACIÓN", true)
);

form.appendChild(
    crearSelect("tablet", "tablet", [{ value: "", text: "Seleccionar Tablet" }], "TABLET")
);

async function update_ayudantes() {
    try {
        const res = await fetch("/get_users");
        const data = await res.json();

        let usuarios = [];

        if (Array.isArray(data)) {
            usuarios = data;
        } else if (typeof data === "object" && data !== null) {
            usuarios = Object.values(data);
        }

        usuarios = usuarios
            .filter(u => u && u.id && u.nombre_completo)
            .sort((a, b) =>
                a.nombre_completo.localeCompare(b.nombre_completo)
            );

        const sel1 = $("#ayudante1");
        const sel2 = $("#ayudante2");

        const selected1 = sel1.value;
        const selected2 = sel2.value;

        sel1.innerHTML = `<option value="">Seleccionar Ayudante 1</option>`;
        sel2.innerHTML = `<option value="">Seleccionar Ayudante 2</option>`;

        usuarios.forEach(user => {
            const uid = String(user.id);

            // Ayudante 1
            if (uid !== String(RESPONSABLE_FINAL) && uid !== selected2) {
                const o1 = document.createElement("option");
                o1.value = uid;
                o1.textContent = user.nombre_completo;
                if (uid === selected1) o1.selected = true;
                sel1.appendChild(o1);
            }

            // Ayudante 2
            if (uid !== String(RESPONSABLE_FINAL) && uid !== selected1) {
                const o2 = document.createElement("option");
                o2.value = uid;
                o2.textContent = user.nombre_completo;
                if (uid === selected2) o2.selected = true;
                sel2.appendChild(o2);
            }
        });

    } catch (err) {
        console.error("Error cargando ayudantes:", err);
        showToast("Error cargando ayudantes", false);
    }
}
update_ayudantes();

$("#ayudante1").onchange = update_ayudantes;
$("#ayudante2").onchange = update_ayudantes;

/* =========================
   NOTAS
========================= */
const notasDiv = document.createElement("div");

const notasLabel = document.createElement("label");
notasLabel.htmlFor = "notas";
notasLabel.textContent = "Notas";
notasDiv.appendChild(notasLabel);

const notasTextArea = document.createElement("textarea");
notasTextArea.id = "notas";
notasTextArea.name = "notas";
notasTextArea.placeholder = "Comentarios adicionales sobre la tarea (máx. 200 caracteres)";
notasTextArea.rows = 4;
notasTextArea.maxLength = 200;
notasDiv.appendChild(notasTextArea);

form.appendChild(notasDiv);

/* =========================
   IMAGENES
========================= */

async function compressImage(file, maxSizeKB = 100) {
    console.log("📸 Imagen original:", {
        name: file.name,
        sizeKB: (file.size / 1024).toFixed(2),
        type: file.type
    });

    return new Promise((resolve, reject) => {
        const img = new Image();
        const reader = new FileReader();

        reader.onload = e => {
            img.src = e.target.result;
        };

        img.onload = () => {
            console.log("🖼️ Imagen cargada:", img.width, "x", img.height);

            let canvas = document.createElement("canvas");
            let ctx = canvas.getContext("2d");

            let width = img.width;
            let height = img.height;

            canvas.width = width;
            canvas.height = height;

            ctx.drawImage(img, 0, 0, width, height);

            let quality = 0.9;

            const tryCompress = () => {
                canvas.toBlob(blob => {
                    if (!blob) {
                        console.error("❌ Blob nulo");
                        return reject();
                    }

                    const sizeKB = blob.size / 1024;
                    console.log(`🔧 Compresión → calidad ${quality.toFixed(2)} | tamaño ${sizeKB.toFixed(2)} KB`);

                    if (sizeKB <= maxSizeKB || quality <= 0.3) {
                        console.log("✅ Imagen final:", {
                            name: file.name,
                            finalSizeKB: sizeKB.toFixed(2)
                        });

                        resolve(new File([blob], file.name, { type: "image/jpeg" }));
                    } else {
                        quality -= 0.1;
                        width *= 0.9;
                        height *= 0.9;

                        canvas.width = width;
                        canvas.height = height;
                        ctx.drawImage(img, 0, 0, width, height);

                        tryCompress();
                    }
                }, "image/jpeg", quality);
            };

            tryCompress();
        };

        reader.onerror = reject;
        reader.readAsDataURL(file);
    });
}

/* ====== DOM (idéntico a tu original) ====== */

const imagenesDiv = document.createElement("div");
imagenesDiv.classList.add("imagenes-container");

const imagenesLabel = document.createElement("label");
imagenesLabel.textContent = "Imágenes";
imagenesDiv.appendChild(imagenesLabel);

const dropArea = document.createElement("div");
dropArea.classList.add("drop-area");
dropArea.textContent = "Haz clic o arrastra imágenes aquí";
imagenesDiv.appendChild(dropArea);

const imagenesInput = document.createElement("input");
imagenesInput.type = "file";
imagenesInput.id = "imagenes";
imagenesInput.name = "imagenes";
imagenesInput.accept = ".png, .jpg, .jpeg";
imagenesInput.multiple = true;
imagenesInput.style.display = "none";

imagenesDiv.appendChild(imagenesInput);
form.appendChild(imagenesDiv);

/* ====== Eventos (idénticos al original) ====== */

dropArea.addEventListener("click", () => {
    imagenesInput.click();
});

imagenesInput.addEventListener("change", async (e) => {
    const files = Array.from(e.target.files);

    console.log("📂 Archivos seleccionados:", files.length);

    if (files.length > 5) {
        showToast("❌ Máximo 5 imágenes", false);
        imagenesInput.value = "";
        return;
    }

    const dt = new DataTransfer();
    dropArea.textContent = "⏳ Procesando imágenes...";

    for (const file of files) {
        if (!file.type.startsWith("image/")) continue;

        const compressed = await compressImage(file, 100);
        dt.items.add(compressed);
    }

    imagenesInput.files = dt.files;

    console.log("📦 Archivos finales en input:", imagenesInput.files);

    dropArea.classList.add("has-files");
    dropArea.textContent = `${imagenesInput.files.length} imagen(es) seleccionada(s)`;
});

dropArea.addEventListener("dragover", (e) => {
    e.preventDefault();
    dropArea.classList.add("dragging");
});

dropArea.addEventListener("dragleave", () => {
    dropArea.classList.remove("dragging");
});

dropArea.addEventListener("drop", async (e) => {
    e.preventDefault();
    dropArea.classList.remove("dragging");

    const files = Array.from(e.dataTransfer.files);

    console.log("📂 Archivos arrastrados:", files.length);

    if (files.length > 5) {
        showToast("❌ Máximo 5 imágenes", false);
        return;
    }

    const dt = new DataTransfer();
    dropArea.textContent = "⏳ Procesando imágenes...";

    for (const file of files) {
        if (!file.type.startsWith("image/")) continue;

        const compressed = await compressImage(file, 100);
        dt.items.add(compressed);
    }

    imagenesInput.files = dt.files;

    console.log("📦 Archivos finales en input:", imagenesInput.files);

    dropArea.classList.add("has-files");
    dropArea.textContent = `${imagenesInput.files.length} imagen(es) seleccionada(s)`;
});

/* =========================
   SUBMIT
========================= */
const submit = document.createElement("button");
submit.textContent = "Aplicar asignación";
submit.className = "btn-submit";
form.appendChild(submit);

form.addEventListener("submit", async (e) => {
    e.preventDefault(); 

    const formData = new FormData(form); 
    formData.append("responsable", RESPONSABLE_FINAL);

    submit.disabled = true;
    submit.textContent = "⏳ Aplicando...";

    try {
        if (!navigator.onLine) {
            showToast("❌ No hay conexión a internet", false);
            return;
        }
        const response = await fetch(form.action, {
            method: form.method, 
            body: formData
        });

        const result = await response.json();

        if (!response.ok) {
            showToast(result.message || "❌ Error al aplicar asignación", false);
            return;
        }

        submit.disabled = false;
        submit.textContent = "Aplicar asignación";

        showToast("✅ Asignación aplicada correctamente", true);

        console.log("Formulario enviado correctamente:", result);

        form.reset();
        dropArea.textContent = "Haz clic o arrastra imágenes aquí";
        dropArea.classList.remove("has-files");

        const modal = document.createElement("div");
        Object.assign(modal.style, {
            position: "fixed",
            top: "50%",
            left: "50%",
            transform: "translate(-50%, -50%)",
            background: "white",
            padding: "20px",
            borderRadius: "10px",
            boxShadow: "0 0 15px rgba(0,0,0,0.3)",
            zIndex: "100000",
            maxHeight: "80vh",
            overflowY: "auto",
            width: "90%",
            maxWidth: "500px",
            boxSizing: "border-box",
        });

        function formatDate(fechaStr) {
            const d = new Date(fechaStr);
            const dd = String(d.getDate()).padStart(2, "0");
            const mm = String(d.getMonth() + 1).padStart(2, "0");
            const yy = String(d.getFullYear());
            return `${dd}-${mm}-${yy}`;
        }

        modal.innerHTML = `
            <h3>Resumen Asignación</h3>
            <div id="modal-content">
            <p><strong>Faena:</strong> ${result.faena || "-"}</p>
            <p><strong>Fecha:</strong> ${formatDate(result.fecha_inicio)}</p>
            <p><strong>Ubicación:</strong> ${result.ubicacion || "-"}</p>
            
            ${result.reemplazo_mmr ? (() => {
                const mmrEntrante = result.reemplazo_mmr?.trim();
                const mmrSaliente = result.mmr?.trim();

                if (!mmrSaliente) {
                    return `<p><strong>MMR entrante:</strong> ${mmrEntrante}</p>`;
                }

                if (mmrEntrante !== mmrSaliente) {
                    return `
                        <p><strong>MMR entrante:</strong> ${mmrEntrante}</p>
                        <p><strong>MMR saliente:</strong> ${mmrSaliente}</p>
                    `;
                }

                return "";
            })() : ""}

            ${result.reemplazo_tablet ? (() => {
                const tabletEntrante = result.reemplazo_tablet?.trim();
                const tabletSaliente = result.tablet?.trim();

                if (!tabletSaliente) {
                    return `<p><strong>Tablet entrante:</strong> ${tabletEntrante}</p>`;
                }

                if (tabletEntrante !== tabletSaliente) {
                    return `
                        <p><strong>Tablet entrante:</strong> ${tabletEntrante}</p>
                        <p><strong>Tablet saliente:</strong> ${tabletSaliente}</p>
                    `;
                }

                return "";
            })() : ""}

            <p><strong>Encargado:</strong> ${result.responsable || "-"}</p>
            </div>
            <div style="margin-top:10px; display:flex; gap:10px; justify-content:center;">
                <button id="copy-json-modal" style="padding:8px 20px;">Copiar</button>
                <button id="close-json-modal" style="padding:8px 20px;">Cerrar</button>
            </div>
        `;

        document.body.appendChild(modal);

        $("#close-json-modal").onclick = () => {
            modal.remove();
        };
        
        $("#copy-json-modal").onclick = () => {
            const container = document.getElementById("modal-content");

            const text = Array.from(container.querySelectorAll("p"))
                .map(p => p.innerText.trim())
                .filter(Boolean)
                .join("\n");

            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(text)
                    .then(() => showToast("✅ Datos copiados al portapapeles", true))
                    .catch(() => showToast("❌ Error al copiar", false));
            } else {
                const textarea = document.createElement("textarea");
                textarea.value = text;
                document.body.appendChild(textarea);
                textarea.select();
                try {
                    document.execCommand("copy");
                    showToast("✅ Datos copiados al portapapeles", true);
                } catch {
                    showToast("❌ Error al copiar", false);
                }
                document.body.removeChild(textarea);
            }
        };

    } catch (err) {
        console.error("Error enviando el formulario:", err);
        showToast("❌ Error al aplicar asignación", false);
    }
});


/* =========================
   FAENAS / UBICACIONES
========================= */
async function cargarFaenas() {
    const res = await fetch("/get_faenas");
    const json = await res.json();
    const sel = $("#faena");

    sel.innerHTML = `<option value="">Seleccionar Faena</option>`;

    Object.keys(json).forEach(f => {
        const o = document.createElement("option");
        o.value = f;
        o.textContent = f;
        sel.appendChild(o);
    });

    sel.onchange = () => {
    const ubic = $("#ubicacion");

    ubic.innerHTML = `<option value="">Seleccionar Ubicación</option>`;

    const ubicaciones = json[sel.value] || [];
    ubicaciones
        .sort((a, b) => a.faenaubicacion.localeCompare(b.faenaubicacion))
        .forEach(u => {
            const option = document.createElement("option");
            option.value = u.faenaubicacion;
            option.textContent = u.faenaubicacion;
            ubic.appendChild(option);
        });
    };
}


cargarFaenas();

/* =========================
   DATALOGGERS
========================= */
async function cargarTablets() {
    try {
        const res = await fetch("/get_dataloggers");
        const data = await res.json();

        const select = document.querySelector("#tablet");
        select.innerHTML = `<option value="">Seleccionar Tablet</option>`;

        const tabletsDisponibles = Object.entries(data)
            .filter(([name, info]) => 
                name.toLowerCase().startsWith("tablet") && info.state === "Disponible"
            )
            .map(([name]) => name) 
            .sort((a, b) => a.localeCompare(b)); 

        tabletsDisponibles.forEach(name => {
            const opt = document.createElement("option");
            opt.value = name;
            opt.textContent = name;
            select.appendChild(opt);
        });

    } catch (err) {
        console.error("Error cargando tablets:", err);
        showToast("Error cargando tablets", false);
    }
}

cargarTablets();

/* =========================
   ACTUALIZAR
========================= */
const btnActualizar = document.createElement("button");
btnActualizar.textContent = "🔄 Actualizar";
btnActualizar.onclick = async ()=>{
    try{
        btnActualizar.disabled=true;
        await fetch("/actualizar",{method:"POST"});
        showToast("Datos actualizados");
    }catch{
        showToast("Error actualización",false);
    }finally{
        btnActualizar.disabled=false;
    }
};
app.appendChild(btnActualizar);
