import threading
from serial import Serial
from time import time, sleep
from lib.utils import Utils


class SerialLib(Utils):
    def __init__(self, usbdevnode, config, baudrate: int = 9600, log_id="SERIAL") -> None:
        super().__init__(log_id)
        self.config = config
        self.baudrate = baudrate
        self.timeout = 0.5
        self.usbdevnode = usbdevnode
        self.limits_pm100 = config["DUST_SENSOR"]["MIN_VALUE_PM100"]
        self.limits_pm10 = config["DUST_SENSOR"]["MIN_VALUE_PM10"]
        self.last_dust_timestamp = time()

        threading.Thread(target=self.connect, daemon=True).start()

    def connect(self) -> None:
        """Attempts to establish a serial connection with the specified USB device node."""
        try:
            self.log(f"Trying to connect to serial port: {self.usbdevnode.get_devnode()}")
            self.serial_module = Serial(self.usbdevnode.get_devnode(), self.baudrate, timeout=self.timeout)
            self.read()
        except Exception as e:
            self.log(f"Error connecting to serial port: {e}")

    def read(self) -> None:
        """Continuously reads and processes lines from the serial module."""
        self.log("Starting to read data from serial port")
        while True:
            try:
                raw_line = self.serial_module.readline().decode("utf-8").strip()
                if raw_line:
                    self.log(f"Received: {raw_line}")
                    self.process_line(raw_line)
            except Exception as e:
                self.log(f"Error reading from serial port: {e}")
            sleep(0.01)

    def process_line(self, line: str) -> None:
        """Processes a received line, identifying the type of data and handling it accordingly."""
        try:
            if "PM100"  in line:
                #  Multiply by 10 to convert from 0.1 µg/m³ to µg/m³
                self.process_dust("pm100", float(line.split(":")[1])*10)
            elif "PM10" in line:
                self.process_dust("pm10", float(line.split(":")[1]))
            else:
                print(f"Unrecognized data type: {line}")
        except Exception as e:
            self.log(f"Error processing line: {line} | Exception: {e}")

    def process_dust(self, pm_type: str, value: float) -> None:
        """Processes particulate matter (PM10, PM100) data, validates and emits it if valid."""
        try:
            limits = self.limits_pm10 if pm_type == "pm10" else self.limits_pm100 if pm_type == "pm100" else None
            if limits < value:
                data = {pm_type: value, "timestamp": int(time())}
                self.log(f"Valid {pm_type.upper()} data: {data}")
                self.emit(data_type=self.log_id, data=data)
                self.last_dust_timestamp = time()
            else:
                self.log(f"{pm_type.upper()} value {value} out of range.")
        except Exception as e:
            self.log(f"Error processing {pm_type} data: {value} | Exception: {e}")


    def filter_data(self, value: float, pm_type: str) -> tuple:
        """Validates particulate matter data based on defined limits."""
        limits = self.limits_pm10 if pm_type == "pm10" else self.limits_pm100 if pm_type == "pm100" else None
        if not limits:
            self.log(f"Invalid PM type: {pm_type}")
            return None, False
        return (value, True) if limits < value else (None, False)
